// escher web sketch:
//
// Copyright 1996, 1997, 1998 by Wes Hardaker and the University of
// Lausanne, Switzerland.  All rights reserved.
//
// Permission to use, copy, modify, and distribute this software and
// its documentation for any purpose and without fee is hereby granted
// for non commerical use, provided that both the above copyright
// notice and this permission notice appear unmodified and intact.
//
// Questions regarding pricing for commerical usage should be sent to
// escher@sphysdec1.unil.ch.
//

import java.applet.Applet;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.geom.Point2D;
import java.awt.image.*;
import java.net.*;
import java.util.Vector;
import java.io.*;
import javax.swing.*;
import javax.vecmath.Point2d;


interface PreviewPainter {
 	public void paintPreview(Graphics g, Dimension d);
}



class PreviewBox extends JComponent {
	PreviewPainter painter;
	
	public PreviewBox(PreviewPainter painter, Dimension size) {
		setPainter(painter);

		setPreferredSize(size);
		setMinimumSize(size);
		setMaximumSize(size);
	}
	
	public void setPainter(PreviewPainter painter) {
		this.painter=painter;
	}

	public void paint(Graphics g) {
		painter.paintPreview(g, getSize());
	}
}


class etool implements AdjustmentListener, ItemListener, PreviewPainter, ButtonPainter {
   // simple point drawing tool
   int dwidth, dheight, width, height;
   int y;
   
   boolean penSizeable;
   esquare canvas;
   JPanel toolchest; //, p;
   int pensize = 1;
   JScrollBar penSizeBar;
   Color color = Color.black;
   Color baseColor = color;
   Color bkgColor = Color.white;
   Color bkgBaseColor = bkgColor;
   int penLum=99;
   int bkgLum=0;
   
   JScrollBar penLightBar;
   JScrollBar bkgLightBar;
   
   colorChoice choice;
   colorChoice bkgChoice;
   
   PreviewBox preview;
   
   //JLabel debugX, debugR1, debugR2;

   
   public void importParams(etool src) {
   	color=src.color;
   	baseColor=src.baseColor;
   	bkgColor=src.bkgColor;
   	bkgBaseColor=src.bkgBaseColor;
   	penLum=src.penLum;
   	bkgLum=src.bkgLum;
   	pensize=src.pensize;
   	if (choice!=null && src.choice!=null) choice.select(src.choice.getSelectedItem());
   	if (bkgChoice!=null && src.bkgChoice!=null) bkgChoice.select(src.bkgChoice.getSelectedItem());
   	if (penSizeBar!=null && src.penSizeBar!=null) penSizeBar.setValue(src.penSizeBar.getValue());
   	if (penLightBar!=null && src.penLightBar!=null) penLightBar.setValue(src.penLightBar.getValue());
   	if (bkgLightBar!=null && src.bkgLightBar!=null) bkgLightBar.setValue(src.bkgLightBar.getValue());
   }
   
   
   public etool() {
   	 penSizeable=true;
		 width = dwidth = 0;
     height = dheight = 0;
   }
   
   public etool(JPanel tools) {
     this();
     toolchest = tools;
     populate_panel();
   }

   public etool(esquare es, int wid, int hei, JPanel p) {
  	 penSizeable=true;
     setSize(wid,hei);
     setCanvas(es);
     toolchest = p;
     populate_panel();
   }

   
   public void populate_panel() {
     JPanel p = new JPanel();
     
     BoxLayout bl = new BoxLayout(p, BoxLayout.Y_AXIS);
     p.setLayout(bl);
     p.setAlignmentX(JPanel.LEFT_ALIGNMENT);
		 
	   p.add(new JLabel(" "));
     p.add(new JLabel("Pen Size"));
	   penSizeBar = new JScrollBar(JScrollBar.HORIZONTAL, pensize, 1, 1, 15);
	   if (!penSizeable) penSizeBar.setEnabled(false);
	   penSizeBar.addAdjustmentListener(this);
	   penSizeBar.setAlignmentX(JPanel.LEFT_ALIGNMENT);
	   p.add(penSizeBar);

	   p.add(new JLabel(" "));
	   p.add(new JLabel("Pen Color"));
	   choice = new colorChoice(baseColor);
     choice.addItemListener(this);
     choice.setAlignmentX(JPanel.LEFT_ALIGNMENT);
		 p.add(choice);

	   p.add(new JLabel(" "));
	   p.add(new JLabel("Pen Brightness"));
     penLightBar = new JScrollBar(JScrollBar.HORIZONTAL, penLum, 1, 0, 100);
     penLightBar.addAdjustmentListener(this);
     penLightBar.setAlignmentX(JPanel.LEFT_ALIGNMENT);
     p.add(penLightBar);
		 
	   p.add(new JLabel(" "));
     p.add(new JLabel("Background color"));
     bkgChoice = new colorChoice(bkgBaseColor);
     bkgChoice.addItemListener(this);
     bkgChoice.setAlignmentX(JPanel.LEFT_ALIGNMENT);
     p.add(bkgChoice);
     
	   p.add(new JLabel(" "));
     p.add(new JLabel("Bg brightness"));
     bkgLightBar = new JScrollBar(JScrollBar.HORIZONTAL, bkgLum, 1, 0, 100);
     bkgLightBar.addAdjustmentListener(this);
     bkgLightBar.setAlignmentX(JPanel.LEFT_ALIGNMENT);
     p.add(bkgLightBar);

	   p.add(new JLabel(" "));

	   JPanel previewCaps = new JPanel();
	   preview = new PreviewBox(this, new Dimension(30, 30));
	   previewCaps.setAlignmentX(JPanel.LEFT_ALIGNMENT);
	   preview.setAlignmentX(JPanel.CENTER_ALIGNMENT);
	   previewCaps.add(preview);
	   p.add(previewCaps);

/*
	   p.add(new JLabel(" "));
	   p.add(debugX=new JLabel("x y"));
	 	 p.add(debugR1=new JLabel("x y"));
		 p.add(debugR2=new JLabel("x y"));
*/	   

     
     toolchest.removeAll();
               
     toolchest.setLayout(new BorderLayout());
     toolchest.add("North", p);
     toolchest.validate();
     toolchest.repaint();
   }


   
   
   
   
   
   public Color getColor() {
     return color;
   }

   public String getColorName() {
     if (choice != null) 
       return (String)choice.getSelectedItem();
     else
       return "black";
   }

   public void setColor(Color c) {
     color = c;
     preview.repaint();
   }
   
   public void setColorName(String c) {
     if (choice != null && choice.getItemCount() > 0 && c != null) {
       choice.select(c);
     } 
   }
   
   
   public static Color colorBright(Color c, double x) {
		if (x==0.0) return c;
		else if (x>0.0) {
			x=1.0-x;
			int r=(int)Math.round(c.getRed()*x);
			int g=(int)Math.round(c.getGreen()*x);
			int b=(int)Math.round(c.getBlue()*x);
			return new Color(r, g, b);
		}
		else {
			x=1.0+x;
			int r=(int)Math.round(255-((255-c.getRed())*x));
			int g=(int)Math.round(255-((255-c.getGreen())*x));
			int b=(int)Math.round(255-((255-c.getBlue())*x));
			return new Color(r, g, b);
		}
   }

   
   public void adjustmentValueChanged(AdjustmentEvent e) {
   	if (e.getSource()==penSizeBar) {
 		 setPenSize(penSizeBar.getValue());
	   return;
   	}
 	
	 	if (e.getSource()==penLightBar) {
	 		penLum = penLightBar.getValue();
	 		if (baseColor.equals(Color.black))
	 			color = colorBright(baseColor, (penLum-100)/160.0);
			else if (baseColor.equals(Color.white))
				color = colorBright(baseColor, (penLum)/160.0);
	 		else color = colorBright(baseColor, (penLum-50)/80.0);
			preview.repaint();
		  return;
	 	}
 	
	 	if (e.getSource()==bkgLightBar) {
	 		bkgLum = bkgLightBar.getValue();
	 		if (bkgBaseColor.equals(Color.black))
	 			bkgColor = colorBright(bkgBaseColor, (bkgLum-100)/160.0);
			else if (bkgBaseColor.equals(Color.white))
				bkgColor = colorBright(bkgBaseColor, (bkgLum)/160.0);
	 		else bkgColor = colorBright(bkgBaseColor, (bkgLum-50)/80.0);
	 		if (preview!=null) preview.repaint();
			if (canvas!=null) {
				if (canvas.undoWasBgBrightChange==false) {
					canvas.setUndoPoint();
					canvas.undoWasBgBrightChange=true;
				}
				else canvas.undoButton.setEnabled(true);
				canvas.erase();
			}
		  return;
	 	}
  }
   
   
   public void itemStateChanged(ItemEvent e) {
   	if (e.getStateChange()==ItemEvent.SELECTED) return;
   	//System.out.println(e);
   	
   	if (e.getSource()==choice) {
   		Color c = colorChoice.stringToColor((String)choice.getSelectedItem()); 
   		if (c!=null) {
   			color = c;
   			baseColor = color;

   			if (c.equals(Color.black)) penLightBar.setValue(99);
   			else if (c.equals(Color.white)) penLightBar.setValue(0);
   			else penLightBar.setValue(50); 
   			penLum = penLightBar.getValue();
   			preview.repaint();
   		  return;
   		}
   	}
 	
   	if (e.getSource()==bkgChoice) {
   		//Color oldColor=bkgColor;
   		Color c = colorChoice.stringToColor((String)bkgChoice.getSelectedItem()); 
   		if (c!=null) {
   			int oldValue=bkgLightBar.getValue();
   			
   			bkgColor = c;
   			bkgBaseColor = bkgColor;
   			
   			if (c.equals(Color.black)) bkgLightBar.setValue(99);
   			else if (c.equals(Color.white)) bkgLightBar.setValue(0);
   			else bkgLightBar.setValue(50); 
   			bkgLum=bkgLightBar.getValue();
   			//repaint();
   			
   			//System.out.println(oldColor+" "+bkgColor);
   			
   			if (canvas!=null && oldValue==bkgLum) {
					canvas.setUndoPoint();
					canvas.erase();
				}
   			return;
   		}
   	}   	
   }


   public void setPenSize(int sz) {
     pensize = sz;
     if (penSizeBar != null)
     	penSizeBar.setValue(sz);
 		preview.repaint();
   }

   public int getPenSize() {
     return pensize;
   }


   
   
   public void setCanvas(esquare es) {
     canvas = es;
   }
   
   public void setSize(int wid, int hei) {
     width = wid;
     dwidth = width*2;
     height = hei;
     dheight = height*2;
   }
   
   public boolean mouseDown(Event evt, int x, int y, Graphics g) {
		 if (g!=null) doit(x,y,g);
     return true;
   }
   
   public boolean mouseDrag(Event evt, int x, int y, Graphics g) {
     if (g!=null) doit(x,y,g);
     return true;
   }
   
   public boolean mouseUp(Event evt, int x, int y, Graphics g) {
		 if (g!=null) doit(x,y,g);
     return true;
   }


   public boolean mouseMove(Event evt, int x, int y, Graphics g) {
		 //if (debugX!=null) debugX.setText(" ( "+x+" "+y+" )");
		 toolchest.repaint();
	   return true;
   }

   
   public void doit(int x, int y, Graphics g) {
     g.setColor(color);
     doit_where(x, y, g);
   }
      
   public void doit_where(int x, int y, Graphics g) {
     int nx, ny;
     int angle=0;
     //System.out.print("x:"+x+" y:"+y);
     Dimension d = canvas.map(x,y);
     x = d.width;
     y = d.height;
		 //System.out.print("  xm:"+x+" ym:"+y);
     int pts[] = canvas.translate_points(x,y);
     int ang[] = canvas.get_angles();
		 int offmod = canvas.getOffset()%dwidth;
     for (int i = 1; i < pts[0]; i = i+2) {
       d = canvas.map(pts[i],pts[i+1]);
	   	 nx = d.width;
       ny = d.height;
  	   //System.out.print("  x["+i/2+"]:"+nx+" y["+i/2+"]:"+ny);
       angle = ang[(i/2)%ang[0]+1];
       doithere(nx, ny, angle, g);
       doithere(nx+dwidth, ny, angle, g);
       doithere(nx+offmod, ny+dheight, angle, g);
       doithere(nx+dwidth+offmod, ny+dheight, angle, g);
       doithere(nx+dwidth-offmod, ny-dheight, angle, g);
       doithere(nx-dwidth, ny, angle, g);
       doithere(nx-offmod, ny-dheight, angle, g);
       doithere(nx-dwidth-offmod, ny-dheight, angle, g);
       doithere(nx-dwidth+offmod, ny+dheight, angle, g);
   //     g.drawString("" + ((i-1)/2+1),nx,ny);
     }
	//System.out.println();
   }
   
   public void doithere(int x, int y, int ang, Graphics g) {
     doithere(x,y,g);
   }
   
   public void doithere(int x, int y, Graphics g) {
     if (pensize == 1)
       g.drawLine(x,y,x,y);
     else
       g.fillOval(x - (int) Math.round(pensize/2),
                  y - (int) Math.round(pensize/2),pensize,pensize);
   }
   
   public void paintButton(Graphics g, int dx, int dy) {
     int p=4;
     g.setColor(Color.black);
     g.fillOval(dx/2-p/2,dy/2-p/2, p, p);
   }



   public void paintPreviewBg(Graphics g, Dimension d) {
   	g.setColor(bkgColor);
   	g.fillRect(0, 0, d.width-1, d.height-1);
    g.setColor(Color.black);
   	g.drawRect(0, 0, d.width-1, d.height-1);
   }
   
   public void paintPreview(Graphics g, Dimension d) {
   	paintPreviewBg(g, d);
   	g.setColor(color);
   	if (pensize == 1) {
      g.drawLine(d.width/2, d.height/2, d.width/2, d.height/2);
    } else {
      g.fillOval(d.width/2-pensize/2,d.height/2-pensize/2, pensize, pensize);
    }
   }

/*   
   public Dimension size() {
   	return new Dimension(30, 30);
   }
   
   
   public Dimension minimumSize() {
     return new Dimension(30,30);
   }
*/

}




class colorChoice extends JComboBox {
	static String[] names = {"black", "blue", "cyan", "darkGray", "gray", "green", "lightGray", 
									 "magenta", "orange", "pink", "red", "white", "yellow", "purple", "indigo"};
	static Color[] colors = {Color.black, Color.blue, Color.cyan, Color.darkGray, 
									 Color.gray, Color.green, Color.lightGray, Color.magenta, 
									 Color.orange, Color.pink, Color.red, Color.white, Color.yellow, new Color(128,0,128), new Color(75,0,130)};
	
	public colorChoice() {
		super();
		for (int i=0; i<names.length; i++) 
			addItem(names[i]);
	}

	public colorChoice(String def) {
		this();
		select(def);
	}
	

	public void select(Object o) {
		setSelectedItem(o);
	}

	
	public colorChoice(Color def) {
		this(colorToString(def));
	}

	static Color stringToColor(String s) {
		for (int i=0; i<names.length; i++) 
			if (s.equals(names[i]))
				return colors[i];
		return null;	
	}

	static String colorToString(Color c) {
		for (int i=0; i<colors.length; i++) 
			if (c.equals(colors[i]))
				return names[i];
		return null;	
	}
}


class drawtool extends etool {
   int lx, ly;
      
   public drawtool(JPanel tools) {
     super(tools);
   }
    
   // override to set lx,ly
   public boolean mouseDown(Event evt, int x, int y, Graphics g) {
     lx = x;
     ly = y;
     doit(x,y,g);
     return true;
   }
   
   public void paintButton(Graphics g, int dx, int dy) {
     Dimension d = new Dimension(dx, dy);
     g.setColor(Color.black);
   	
     int x6 = d.width/8;
     int y6 = d.height/10;
     g.translate(d.width/6, d.height/4);
     
     g.drawLine(5*x6,5*y6,3*x6,3*y6);
     g.drawLine(3*x6,3*y6,2*x6,2*y6);
     g.drawLine(2*x6,y6,2*x6,2*y6);
     g.drawLine(3*x6,y6/2,2*x6,y6);
     g.drawLine(3*x6,y6/2,4*x6,y6);
     g.drawLine(4*x6,2*y6,4*x6,y6);
     g.drawLine(4*x6,2*y6,3*x6,3*y6);
     g.drawLine(3*x6,3*y6,x6,5*y6);
     g.translate(-d.width/6, -d.height/6);
   }

   public void paintPreview(Graphics g, Dimension d) {
   	paintPreviewBg(g, d);
   	g.setColor(color);
    Graphics2D g2d = (Graphics2D)g;
 	  g2d.setStroke(new BasicStroke((float) pensize)); 
 	  g2d.drawLine(d.width/4, d.height/2, 3*d.width/4, d.height/2);
  }
   
   
   public void doit(int x, int y, Graphics g) {
    set_line(x,y,lx,ly,g);
 	  lx = x;
    ly = y;
  }

   
   public void drawLine(int x1, int y1, int x2, int y2, Graphics gr) {
   	Graphics2D g2d = (Graphics2D)gr;
    g2d.setColor(color);
 	  g2d.setStroke(new BasicStroke((float) pensize, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND)); 
 	  g2d.drawLine(x1, y1, x2, y2);
   }

   
   public void set_line(int x1, int y1, int x2, int y2, Graphics gr) {
		if (gr==null) return;

		//set_line2(x1, y1, x2, y2, gr);


		int n2=(int)Math.floor((double)y2/dheight);
		int off=canvas.getOffset();
		int x2t = (x2-(n2*(off%dwidth)))%dwidth;
		if (x2t<0) x2t=dwidth+x2t;
		int lx2=(x2-(n2*(off%dwidth)))-x2t;

		int pts1[] = canvas.translate_points(x1, y1);
		int pts2[] = canvas.translate_points(x2, y2);


		//gr.setColor(Color.black);

    for (int i=1, a=1; i<pts1[0]; i+=2, a++) {
    	set_line2(pts1[i]+(2*lx2), pts1[i+1], pts2[i]+(2*lx2), pts2[i+1], gr);

		//gr.setColor(Color.blue);

    }
   }
   
   
   public void set_line2(int x1, int y1, int x2, int y2, Graphics gr) {

    int n1=(int)Math.floor((double)y1/dheight);
		int n2=(int)Math.floor((double)y2/dheight);
		int off=canvas.getOffset();

		x1 -= (n1*(off%dwidth));
		x2 -= (n2*(off%dwidth));

		int x1t = x1%dwidth;
		int x2t = x2%dwidth;
		int y1t = y1%dheight;
		int y2t = y2%dheight;

		if (x1t<0) x1t=dwidth+x1t;
		if (x2t<0) x2t=dwidth+x2t;
    
    int lx1, lx2;
    if (x2<x1) {
     	if (x2t<pensize/2) lx2 = x2-pensize/2-((x2t+dwidth-pensize/2)%dwidth);
      else lx2 = x2-x2t;
     	if (x1t+pensize/2>dwidth) lx1 = x1+pensize/2-((x1t+pensize/2)%dwidth);
      else lx1 = x1-x1t;
    }
    else {
     	if (x2t+pensize/2>dwidth) lx2 = x2+pensize/2-((x2t+pensize/2)%dwidth);
      else lx2 = x2-x2t;
      if (x1t<pensize/2) lx1 = x1-pensize/2-((x1t+dwidth-pensize/2)%dwidth);
      else lx1 = x1-x1t;
    }
    lx1/=dwidth;
    lx2/=dwidth;
    
    int ly1, ly2;
    if (y2<y1) {
     	if (y2t<pensize/2) ly2 = y2-pensize/2-((y2t+dheight-pensize/2)%dheight);
      else ly2 = y2-y2t;
     	if (y1t+pensize/2>dheight) ly1 = y1+pensize/2-((y1t+pensize/2)%dheight);
      else ly1 = y1-y1t;
    }
    else {
     	if (y2t+pensize/2>dheight) ly2 = y2+pensize/2-((y2t+pensize/2)%dheight);
      else ly2 = y2-y2t;
      if (y1t<pensize/2) ly1 = y1-pensize/2-((y1t+dheight-pensize/2)%dheight);
      else ly1 = y1-y1t;
    }
    ly1/=dheight;
    ly2/=dheight;
    
		//debugR2.setText("n:"+n2+" "+(x2+(n2*off))+" "+x2t+" ("+(x2-(x2t))+" "+(y2-y2t)+")");
		//debugR1.setText("n:"+n1+" "+(x1+(n1*off))+" "+x1t+" ("+(x1-(x1t))+" "+(y1-y1t)+")");
		//debugR1.setText(n1+" ("+lx2+" "+ly2+")"+" -> ( "+lx1+" "+ly1+")");

		x1 += (n1*(off%dwidth));
		x2 += (n2*(off%dwidth));
    
    
    
    //System.out.println(lx2+"->"+lx1+";"+ly2+"->"+ly1);
    for (int i=((int)Math.min(lx1, lx2)-(off==0?0:1)); i<=((int)Math.max(lx1, lx2)+(off==0?0:1)); i++) {
     	for (int j=(int)Math.min(ly1, ly2); j<=(int)Math.max(ly1, ly2); j++) {
     		//System.out.print("i:"+i+" j:"+j+" | ");
     		drawLine(x1-i*dwidth-j*(off%dwidth), y1-j*dheight, x2-i*dwidth-j*(off%dwidth), y2-j*dheight, gr);
     	}
    }
		//System.out.println();
		
	}
}
   
class linetool extends drawtool {
      
   public linetool(JPanel tools) {
     super(tools);
   }
      
   public void paintButton(Graphics g, int dx, int dy) {
     g.setColor(Color.black);
     g.drawLine(dx/3, dy/3, 2*dx/3, 2*dy/3);
   }
   
   public boolean mouseDrag(Event evt, int x, int y, Graphics gr) {
     canvas.Undo(true);
     doit(x,y,gr);
     return true;
   }
   
   public void doit(int x, int y, Graphics gr) {
     set_line(x,y,lx,ly,gr);
   }

   
   public void drawLine(int x1, int y1, int x2, int y2, Graphics gr) {
   	Graphics2D g2d = (Graphics2D)gr;
    g2d.setColor(color);
 	  g2d.setStroke(new BasicStroke((float) pensize, BasicStroke.CAP_BUTT, BasicStroke.JOIN_ROUND)); 
 	  g2d.drawLine(x1, y1, x2, y2);
   }
   
   
   public boolean mouseUp(Event evt, int x, int y, Graphics gr) {
     if (lx != x || ly != y) {
       return mouseDrag(evt,x,y,gr);
     }
     return true;
   }
}
   
class boxtool extends linetool {
	
   public boxtool(JPanel tools) {
     super(tools);
   }
      
   public void paintButton(Graphics g, int dx, int dy) {
     	g.setColor(Color.black);
      g.drawRect(dx/3, dy/3, dx/3, dy/3);
   }

   public void drawLine(int x1, int y1, int x2, int y2, Graphics gr) {
   	Graphics2D g2d = (Graphics2D)gr;
    g2d.setColor(color);
 	  g2d.setStroke(new BasicStroke((float) pensize, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER)); 
 	  g2d.drawLine(x1, y1, x2, y2);
   }
   
   public void doit(int x, int y, Graphics gr) {
    set_line(x,y,lx,y,gr);
    set_line(x,y,x,ly,gr);
    set_line(lx,ly,lx,y,gr);
    set_line(lx,ly,x,ly,gr);
   }
   
}
   
class filledboxtool extends linetool {
      
   public filledboxtool(JPanel tools) {
     super(tools);
   	 penSizeable=false;
   }
      
   public void paintButton(Graphics g, int dx, int dy) {
     g.setColor(Color.black);
     g.fillRect(dx/3, dy/3, dx/3, dy/3);
   }

   public void paintPreview(Graphics g, Dimension d) {
   	paintPreviewBg(g, d);
   	g.setColor(color);
    g.fillRect(d.width/4, d.height/4, d.width/2, d.height/2);
	}
   
   public void doit(int x, int y, Graphics gr) {
    int opsz = pensize;
    pensize=Math.abs(ly-y);
    int newy = (ly+y)/2;
    set_line(x,newy,lx,newy,gr);
    pensize = opsz;
  }
   
}
   
class imgobject implements ButtonPainter {
   Image rots[] = new Image[1080];
   Canvas parent;
      
   public imgobject(Image i, Canvas p) {
     rots[0] = i;
     parent = p;
   }
   
   public Image get() {
     return rots[0];
   }
   
   public Image get(int ang) {
     if (rots[ang] == null) {
       ImageFilter filter = new RotateFilter(ang);
       ImageProducer prod = new FilteredImageSource(rots[0].getSource(),filter);
       rots[ang] = parent.createImage(prod);
       Toolkit tool = Toolkit.getDefaultToolkit();
       MediaTracker tracker = new MediaTracker(parent);
       tracker.addImage(rots[ang],0);
       
       try {
         tracker.waitForAll();
       } catch (java.lang.InterruptedException e) {
         System.out.println("InterruptedException");
       }
     }
     if (rots[ang] != null) {
       return rots[ang];
     } else {
       return rots[0];
     }
   }

   
   
	 public void paintButton(Graphics g, int sz, int sy) {
		int x = (sz-rots[0].getWidth(null))/2;
		int y = (sy-rots[0].getHeight(null))/2;
	 	
	 	g.drawImage(get(), x, y, null);
	 }
}
   
// derive from linetool since it saves tmpimg until mouseUp.
class imagetool extends linetool implements ImageObserver, ButtonReceiver, ItemListener {
   Image img;
   imgobject io;
   int ix, iy;
   URL url;
   String imgList[][];
   String imgFileList[];
   Image imgs[];
   int numlists;
   int currentlist = 0;
   boolean dorotate = true;
   JCheckBox checkbox;
   JComboBox imgChoice;
   Vector magnet;
   
   
   public imagetool(URL l, Image i, JPanel tools) {
     super(tools);
     magnet = new Vector(110, 10);
     toolchest = tools;
     img = i;
     url = l;
     img = i;
     if (img != null) {
       iy = (int) Math.round(img.getHeight(this)/2);
       ix = (int) Math.round(img.getWidth(this)/2);
     }
     read_lists();
     populate_panel();
   }

   
   public void select(Object selected) {
    io = ((imgobject) selected);
    img = io.get();
    ix = (int) Math.round(img.getWidth(this)/2);
    iy = (int) Math.round(img.getHeight(this)/2);
   }
   
   
   
   public boolean imageUpdate(Image i, int  infoflags,
                              int  x, int  y, int  width, int  height) {
     if ((infoflags & ImageObserver.ALLBITS) != 0) {
       ix = (int) Math.round(img.getWidth(null)/2);
       iy = (int) Math.round(img.getHeight(null)/2);
       return false;
     } else if ((infoflags & ImageObserver.WIDTH) != 0 ||
                (infoflags & ImageObserver.HEIGHT) != 0) {
       ix = (int) Math.round(img.getWidth(this)/2);
       iy = (int) Math.round(img.getHeight(this)/2);
     }
     return true;
   }
      
   public void read_lists() {
     Toolkit tool = Toolkit.getDefaultToolkit();
     MediaTracker tracker = new MediaTracker(toolchest);
     URL u;
     try {
	  	 u = getClass().getResource("imageList.txt");
	  	 if (u==null) u = new URL(url, "imageList.txt");
     } catch (java.net.MalformedURLException e) {
       System.out.println("URL exception");
       return;
     }
     try {
       URLConnection c = u.openConnection();
       DataInputStream is = new DataInputStream(c.getInputStream());
       String s = is.readLine();
       numlists = Integer.parseInt(s);
       imgList = new String[numlists][];
       imgFileList = new String[numlists];
       for(int i=0; i < numlists; i++) {
         imgFileList[i] = is.readLine();
       }
       for(int i=0; i < numlists; i++) {
         try {
	  	  	  u = getClass().getResource(imgFileList[i]);
	  	 	  if (u==null) u = new URL(url, imgFileList[i]);
         } catch (java.net.MalformedURLException e) {
           System.out.println("URL exception");
           return;
         }
         c = u.openConnection();
         is = new DataInputStream(c.getInputStream());
         s = is.readLine();
         int num = Integer.parseInt(s);
         imgList[i] = new String[num];
         for(int j=0; j < num; j++) {
           imgList[i][j] = is.readLine();
         }
       }
     } catch (java.io.IOException e) {
     }
   }

	 
	 public void itemStateChanged(ItemEvent e) {
	 	super.itemStateChanged(e);
	 	
	 	if (e.getSource()==checkbox) {
			dorotate=((JCheckBox)e.getSource()).isSelected();
		}
	 	 if (e.getSource()==imgChoice) {
     for(int i=0; i < numlists; i++) {
       if (imgChoice.getSelectedItem().equals(imgFileList[i])) {
         currentlist = i;
         populate_panel();
         return;
       }
     }
	 	}
	 }

   public void populate_panel() {  	
   	if (img != null && url != null) {
       try {
         Toolkit tool = Toolkit.getDefaultToolkit();
         MediaTracker tracker = new MediaTracker(toolchest);
   
         toolchest.removeAll();
         toolchest.setLayout(new BorderLayout());
         JPanel p1 = new JPanel();
         toolchest.add("Center",p1);
         if (imgList[currentlist].length >= 10)
           p1.setLayout(new GridLayout((imgList[currentlist].length+1)/2,2));
         else
           p1.setLayout(new GridLayout(imgList[currentlist].length,1));
         imgs = new Image[imgList[currentlist].length];
         for (int i = 0; i < imgList[currentlist].length; i++) {
	  	 	  URL uu = getClass().getResource(imgList[currentlist][i]);
	  	 	  if (uu==null) uu = new URL(url, imgList[currentlist][i]);
           imgs[i] = tool.getImage(uu);
           tracker.addImage(imgs[i], i);
         }
         
         try {
           tracker.waitForAll();
         } catch (java.lang.InterruptedException e) {
           System.out.println("InterruptedException");
         }
         

		     ButtonGroup group = new ButtonGroup();
    		 ToolButton[] imgButtons = new ToolButton[imgList[currentlist].length];
         for (int i = 0; i < imgList[currentlist].length; i++) {
           imgButtons[i] = new ToolButton(io=new imgobject(imgs[i], new Canvas()), this);
           group.add(imgButtons[i]);
           p1.add(imgButtons[i]);
         }
         img = imgs[imgList[currentlist].length-1];
         imgButtons[imgList[currentlist].length-1].setSelected(true);
       } catch (java.net.MalformedURLException e) {
         System.out.println("URL exception");
       }
       
	     JPanel p = new JPanel();

       //p.setLayout(new BorderLayout());

	     BoxLayout bl = new BoxLayout(p, BoxLayout.Y_AXIS);
	     p.setLayout(bl);
	     p.setAlignmentX(JPanel.LEFT_ALIGNMENT);


       imgChoice = new JComboBox();
       for(int i=0; i < numlists; i++) {
         imgChoice.addItem(imgFileList[i]);
       }
       imgChoice.setSelectedItem(imgFileList[currentlist]);
       imgChoice.addItemListener(this);

       p.add("South",imgChoice);
       p.add("North",checkbox = new JCheckBox("Rotate", null, dorotate));
       checkbox.addItemListener(this);

     	 imgChoice.setAlignmentX(JPanel.LEFT_ALIGNMENT);
     	 checkbox.setAlignmentX(JPanel.LEFT_ALIGNMENT);




		   p.add(new JLabel(" "));
	
	     p.add(new JLabel("Background color"));
	     bkgChoice = new colorChoice(bkgBaseColor);
	     bkgChoice.addItemListener(this);
	     bkgChoice.setAlignmentX(JPanel.LEFT_ALIGNMENT);
	     p.add(bkgChoice);
	     
	     p.add(new JLabel("Bg brightness"));
	     bkgLightBar = new JScrollBar(JScrollBar.HORIZONTAL, bkgLum, 1, 0, 100);
	     bkgLightBar.addAdjustmentListener(this);
	     bkgLightBar.setAlignmentX(JPanel.LEFT_ALIGNMENT);
	     p.add(bkgLightBar);

       
       
       toolchest.add("South",p);
       toolchest.validate();
       toolchest.repaint();
     }  
   }
   public String getColorName() {
     if (dorotate)
       return "true";
     return null;
   }
         
   public void setColorName(String c) {
     if (c == null)
       dorotate=false;
     else
       dorotate=true;
   }
      
   public void paintButton(Graphics g, int dx, int dy) {
     ix = (int) Math.round(img.getWidth(this)/2);
     iy = (int) Math.round(img.getHeight(this)/2);
     g.drawImage(img,dx/2-ix,dy/2-iy,null);
   }
         
   public void doit(int x, int y, Graphics g) {
   	magnet.clear();
   	doit_where(x, y, g);
   }
         
   public void doithere(int x, int y, int ang, Graphics gr) {
     if (gr==null) return;
     MagnetPoint p = new MagnetPoint(x, y);
   	 if (!magnet.contains(p)) {
       magnet.add(p);
	     if (io == null || !dorotate) {
	       	 gr.drawImage(img,x-ix,y-iy,null);
	     } else {
	       Image i = io.get(ang);
	       int ix = img.getWidth(null);
	       int iy = img.getHeight(null);
	       double a = (ang%90)*Math.PI/180.0;
	       int xoff = (int) Math.round((iy*Math.cos(a) + ix*Math.sin(a))/2);
	       int yoff = (int) Math.round((ix*Math.cos(a) + iy*Math.sin(a))/2);
	       gr.drawImage(i,x-xoff,y-yoff,null);
	     }
     }
   }
   
   
   class MagnetPoint extends Point {
   	 public MagnetPoint(int x, int y) {
   	 	super(x, y);
   	 }
   	 
   	 public boolean equals(Object obj) {
   	 	 return obj instanceof Point2D && Math.abs(distance((Point2D)obj))<4;
   	 }
   }
}
      



