#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""Smooth and reduce point count of contour curve objects within tolerance.
Uses RhinoCommon methods Polyline.ReduceSegments(), Polyline.Smooth(), Curve.Fit().
Replaces original object (same ID), preserves object attributes.
Splines/polycurves will first be converted to polylines using the same fitting
tolerance as the rest.  Outputs polylines unless fit curves option is activated, 
in which case curves are refit to degree 3. Ignores lines, circles, arcs, ellipses.
Script by Mitch Heynick 06.11.17.
- Localized 07.11.17
Revised 26.02.18 - fixed bugs with fit tolerance sticky value and localization
"""


import rhinoscriptsyntax as rs
import Rhino
import scriptcontext as sc
from System import Guid

def Get2DblsPlusIntPlusBool(prompt,ops,inis):
    go = Rhino.Input.Custom.GetOption()
    go.SetCommandPrompt(prompt)
    #initialize
    dblOption0=Rhino.Input.Custom.OptionDouble(*inis[0])
    dblOption1=Rhino.Input.Custom.OptionDouble(*inis[1])
    intOption2=Rhino.Input.Custom.OptionInteger(*inis[2])
    blnOption3=Rhino.Input.Custom.OptionToggle(*inis[3])
    #add options
    go.AddOptionDouble(ops[0],dblOption0)
    go.AddOptionDouble(ops[1],dblOption1)
    go.AddOptionInteger(ops[2],intOption2)
    go.AddOptionToggle(ops[3],blnOption3)
    go.AcceptNothing(True)
    #get
    while True:
        get_rc = go.Get()
        if get_rc==Rhino.Input.GetResult.Cancel: return
        elif get_rc==Rhino.Input.GetResult.Nothing: break
        elif get_rc==Rhino.Input.GetResult.Option: continue
            
    a=dblOption0.CurrentValue
    b=dblOption1.CurrentValue
    c=intOption2.CurrentValue
    d=blnOption3.CurrentValue
    return (a,b,c,d)

def plpcs_filt(rhino_object, geometry, component_index):
    #polylines, polycurves, splines
    a=rs.IsPolyline(geometry) and not rs.IsLine(geometry)
    b=rs.IsPolyCurve(geometry)
    c=not(rs.IsCircle(geometry) or rs.IsArc(geometry))
    d=not(rs.IsEllipse(geometry) or rs.IsLine(geometry))
    return a | b | (c & d)

def Localize(local_id):
    #pass rs.LocaleID() as argument
    msgs=[]
    if local_id==1036:
        msgs.append ("Sélectionner les courbes à traiter")
        a=["ToléranceAdj","FacteurLissage","ItérationsLissage","SortieSplines"]
        msgs.append(a)
        msgs.append("Options de reconstruction")
        msgs.append("Non")
        msgs.append("Oui")
        msgs.append("Calcul...")
        msgs.append("Courbes traitées {}.")
        msgs.append("Nombre de points réduit de {: .0f}%")
        msgs.append("{} courbes ont eu des erreurs dans le traitement")
        msgs.append("{} les courbes n'ont pas itéré à max")
    else:
        msgs.append("Select curves to process")
        a=["FittingTolerance","SmoothFactor","SmoothIterations","OutputSplines"]
        msgs.append(a)
        msgs.append("Reconstruction options")
        msgs.append("No")
        msgs.append("Yes")
        msgs.append("Processing curves")
        msgs.append("Processed {} curves.")
        msgs.append(" Point count reduced by {:.0f}%")
        msgs.append(" {} curves had errors in processing")
        msgs.append("{} curves did not iterate to max")
    return msgs

def CombinedFixContourCrvs():
    msgs=Localize(rs.LocaleID())
    crvIDs=rs.GetObjects(msgs[0],4,preselect=True,custom_filter=plpcs_filt)
    if not crvIDs: return
    #previous settings
    if sc.sticky.has_key("CombFix_Tol"): user_tol = sc.sticky["CombFix_Tol"]
    else: user_tol = 0.1
    if sc.sticky.has_key("CombFix_Smooth"): user_smooth=sc.sticky["CombFix_Smooth"]
    else: user_smooth=0.0
    if sc.sticky.has_key("CombFix_Iter"): user_iter=sc.sticky["CombFix_Iter"]
    else: user_iter=5
    if sc.sticky.has_key("CombFix_Fit"): user_fit=sc.sticky["CombFix_Fit"]
    else: user_fit=False
    
    max_iter=25 #max 25 smoothing iterations
    abs_tol=sc.doc.ModelAbsoluteTolerance
    dbl_ini_0=[user_tol,True,abs_tol] #min abs_tol
    dbl_ini_1=[user_smooth,0,1.0] #between 0 and 1.0
    int_ini_2=[user_iter,0,max_iter] #between 0 and max
    bool_ini_3=[user_fit,msgs[3],msgs[4]]
    inis=[dbl_ini_0,dbl_ini_1,int_ini_2,bool_ini_3]
    
    result=Get2DblsPlusIntPlusBool(msgs[2],msgs[1],inis)
    if not result: return
    tol,smf,sm_iter,fit_crv=result
    
    rs.StatusBarProgressMeterShow(msgs[5],0,len(crvIDs),True,True)
    rept=[len(crvIDs),0,0]
    errors=0
    iter_error=0
    check_iter=0
    for i, crvID in enumerate(crvIDs):
        obj=sc.doc.Objects.Find(crvID)
        if not rs.IsPolyline(crvID):
            #convert non-polyline curve to polyline curve
            crvObj=obj.Geometry.ToPolyline(0,0,0,0,0,tol,abs_tol,0,True)
        else:
            crvObj=obj.Geometry
        if crvObj:
            #convert to polyline object
            rc,pl=Rhino.Geometry.PolylineCurve.TryGetPolyline(crvObj)
            if rc:
                orig_count=pl.Count
                #reduce number of segments
                pl.ReduceSegments(tol)
                #Rhino.Geometry.Polyline.ReduceSegments(pl,tol)
                if pl.IsValid:
                    new_count=pl.Count
                    #smooth reduced polyline
                    if smf>0:
                        j=0
                        while j<sm_iter:
                            #don't smooth if smooth factor (smf)= 0
                            if smf==0: break
                            #print "Iteration {}".format(j)
                            Rhino.Geometry.Polyline.Smooth(pl,smf)
                            j+=1
                            #check validity, continue for next iteration
                            if not pl.IsValid:
                                errors+=1 ; break
                        if j<sm_iter: iter_error+=1
                        #replace original with new
                    if pl.IsValid:
                        rept[1]+=orig_count
                        rept[2]+=new_count
                        #fit curve routine
                        if fit_crv:
                            plc=pl.ToNurbsCurve()
                            if plc:
                                plc.IncreaseDegree(3)
                                pl=plc.Fit(3,0,0)
                        if pl and pl.IsValid:
                            replaced=sc.doc.Objects.Replace(crvID,pl)
                            if not replaced: errors+=1
                        else:
                            errors+=1 ; continue
        rs.StatusBarProgressMeterUpdate(i,True)
    sc.doc.Views.Redraw()
    rs.StatusBarProgressMeterHide()

    #store settings and report
    sc.sticky["CombFix_tol"] = tol
    sc.sticky["CombFix_Smooth"] = smf
    sc.sticky["CombFix_Iter"] = sm_iter
    sc.sticky["CombFix_Fit"] = fit_crv
    
    msg=msgs[6].format(rept[0])
    msg+=msgs[7].format(100-(rept[2]/rept[1])*100)
    if errors>0: msg+=msgs[8].format(errors)
    print msg
    if iter_error>0: print msgs[9].format(iter_error)
CombinedFixContourCrvs()