#include <stdlib.h>
#include <tgmath.h>
#include <string.h>
#include "xil_printf.h"
#include "edram.h"
#include "events.h"
#include "rand.h"

static int drt_event_comp(const void *a, const void *b)
{
    // sort by row, drt
    const drt_event *f = a;
    const drt_event *s = b;
    // if (f->row > s->row)
    //     return 1;
    // else if (f->row < s->row)
    //     return -1;
    // else
    if (f->drt > s->drt)
        return 1;
    else if (f->drt < s->drt)
        return -1;
    else
        return 0;
}

static void events_write_row(unsigned int row, const drt_event *events, unsigned int nevents, int action, int shuffle, unsigned int *evptr)
{
    if (*evptr + nevents > (1 << XPAR_EDRAM_0_NEVENTS_LOG)) {
        xil_printf("Error: Too many events!\n");
        abort();
    }
    if (shuffle)
        row = rand() % (1 << XPAR_EDRAM_0_ROWS_LOG);
    PRAM_SET(row, *evptr, nevents);
    for (unsigned int i = 0; i < nevents; i++) {
        unsigned int col = shuffle ? rand() % (1 << XPAR_EDRAM_0_WORDSIZE_LOG) : events[i].col;
        EVRAM_SET((*evptr)++, events[i].drt, action, col);
    }
}

static void events_write(const drt_event_list *drts, int action, int shuffle)
{
    // qsort(drts->events, drts->nevents, sizeof(drt_event), drt_event_comp);
    unsigned int evptr = 0;
    for (unsigned int offset = 0; offset < (1 << XPAR_EDRAM_0_ROWS_LOG); offset += (1 << drts->rows_log)) {
        unsigned int prev = 0;
        for (unsigned int i = 0; i < drts->nevents; i++) {
            if (drts->events[i].row != drts->events[prev].row) {
                events_write_row(offset + drts->events[prev].row, &drts->events[prev], i - prev, action, shuffle, &evptr);
                prev = i;
            }
        }
        events_write_row(offset + drts->events[prev].row, &drts->events[prev], drts->nevents - prev, action, shuffle, &evptr);
    }
}

static void events_create_rand(drt_event_list *drts, unsigned int max_events, float mu, float sigma, int rfint)
{
    drts->nevents = 0;
    for (int i = 0; i < (1 << drts->rows_log); i++) {
        unsigned int prev = drts->nevents;
        for (unsigned int j = 0; j < (1 << XPAR_EDRAM_0_WORDSIZE_LOG); j += 2) {
            float r[2];
            randn2(&r[0], &r[1]);
            for (int k = 0; k < 2; k++) {
                int drt = exp(mu + r[k] * sigma);
                if (drt <= rfint) {
                    if (drts->nevents >= max_events) {
                        xil_printf("Error: Too many random events!\n");
                        abort();
                    }
                    drts->events[drts->nevents++] = (drt_event) {i, j + k, drt};
                }
            }
        }
        qsort(&drts->events[prev], drts->nevents - prev, sizeof(drt_event), drt_event_comp);
    }
}

void events_reset()
{
    SET_MODSEL(0);
    memset((char *) PRAM_BASE, 0, 4 << XPAR_EDRAM_0_ROWS_LOG);
    // memset((char *) EVRAM_BASE, 0, 4 << XPAR_EDRAM_0_NEVENTS_LOG);
}

void events_init(const drt_event_list *drts, int action, int shuffle)
{
    events_reset();
    if (shuffle) {
        srand(shuffle);
        for (unsigned int i = 0; i < (1 << XPAR_EDRAM_0_ARRAYS_LOG); i++) {
            SET_MODSEL(i + 1);
            events_write(drts, action, shuffle);
        }
    } else {
        SET_MODSEL(0);
        events_write(drts, action, shuffle);
    }
}

void events_init_rand(float mu, float sigma, int rfint, int action, int seed)
{
    events_reset();
    srand(seed);

    drt_event_list *drts = malloc(sizeof(drt_event_list) + (1 << XPAR_EDRAM_0_ROWS_LOG) * sizeof(drt_event));
    drts->rows_log = XPAR_EDRAM_0_ROWS_LOG;
    events_create_rand(drts, 1 << XPAR_EDRAM_0_ROWS_LOG, mu, sigma, rfint);

    for (unsigned int i = 0; i < (1 << XPAR_EDRAM_0_ARRAYS_LOG); i++) {
        SET_MODSEL(i + 1);
        events_write(drts, action, 1);
    }

    free(drts);
}
