#include <stdlib.h>
#include "xil_printf.h"
#include "convolution.h"
#include "edram.h"
#include "events.h"
#include "malloc.h"
#include "types.h"

#include "drt.h"

#define RFINT 20
#define MU 13.0
#define SIGMA 1.7

#define RFSKIP 5
#define MAX_RFINT 64
#define ACTION ACTION_DRT1

static void print_result(const benchmark_result *result, const void *golden)
{
    convolution_print_result(result->result, golden);
    xil_printf("Execution time:\n");
    xil_printf("%d cycles\n", GET_SIMTS());
    xil_printf("\n");
}

static benchmark_result run(unsigned int rfint)
{
    EDRAM_RESET();
    malloc_reset();

    SET_RFSKIP(RFSKIP);
    SET_RFINT(rfint);

    EDRAM_START();
    benchmark_result result = convolution_run();
    EDRAM_STOP();

    return result;
}

int main()
{
    void *golden;
    benchmark_result result;

    // SRAM Reference run (no errors, no refresh)
    xil_printf("REFERENCE\n");
    events_reset();
    result = run(0);
    print_result(&result, NULL);

    // Save reference result
    golden = malloc(result.size);
    if (!golden) {
        xil_printf("malloc for golden failed\n");
        abort();
    }
    memcpy(golden, result.result, result.size);

    // Static DRT map:
    xil_printf("STATIC DRT MAP\n");
    events_init(&drts, ACTION, 0);
    result = run(RFINT);
    print_result(&result, golden);

    // Shuffled DRT map:
    xil_printf("SHUFFLED DRT MAP\n");
    events_init(&drts, ACTION, 1);
    result = run(RFINT);
    print_result(&result, golden);

    // Random DRT map:
    xil_printf("RANDOM DRT MAP\n");
    events_init_rand(MU, SIGMA, MAX_RFINT << (XPAR_EDRAM_0_ROWS_LOG - RFSKIP), ACTION, 1);
    result = run(RFINT);
    print_result(&result, golden);

    xil_printf("Done.");
    free(golden);
}
