library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

library axi_lite_ipif_v3_0_4;
use axi_lite_ipif_v3_0_4.axi_lite_ipif;
use axi_lite_ipif_v3_0_4.ipif_pkg.SLV64_ARRAY_TYPE;
use axi_lite_ipif_v3_0_4.ipif_pkg.INTEGER_ARRAY_TYPE;

entity vivado is
    generic(
        C_FAMILY          : string;

        -- Memory AXI generics
        C_MAXI_AWIDTH     : integer;
        C_MAXI_DWIDTH     : integer;
        C_MAXI_DWIDTH_LOG : integer;

        -- Control AXI generics
        C_CAXI_AWIDTH     : integer;
        C_CAXI_DWIDTH     : integer;
        C_CAXI_DWIDTH_LOG : integer;

        -- User generics
        WORDSIZE_LOG      : natural;
        ARRAYS_LOG        : natural;
        ROWS_LOG          : natural;
        TIMERSIZE         : natural;
        TDIFFSIZE         : natural;
        ACTSIZE           : natural;
        SKIPSIZE          : natural;
        READWRITE         : boolean;
        MEMARRAY_MODEL    : string;
        MOD_MODEL         : string;
        NEVENTS_LOG       : natural;
        PRAM_INITFILE     : string  := "";
        EVRAM_INITFILE    : string  := "";
        DEBUG             : boolean := false
    );
    port(
        -- Memory AXI port
        MAXI_clk     : in  std_logic;
        MAXI_rstn    : in  std_logic;
        MAXI_awaddr  : in  std_logic_vector(C_MAXI_AWIDTH - 1 downto 0);
        MAXI_awvalid : in  std_logic;
        MAXI_awready : out std_logic;
        MAXI_wdata   : in  std_logic_vector(C_MAXI_DWIDTH - 1 downto 0);
        MAXI_wstrb   : in  std_logic_vector((C_MAXI_DWIDTH / 8) - 1 downto 0);
        MAXI_wvalid  : in  std_logic;
        MAXI_wready  : out std_logic;
        MAXI_bresp   : out std_logic_vector(1 downto 0);
        MAXI_bvalid  : out std_logic;
        MAXI_bready  : in  std_logic;
        MAXI_araddr  : in  std_logic_vector(C_MAXI_AWIDTH - 1 downto 0);
        MAXI_arvalid : in  std_logic;
        MAXI_arready : out std_logic;
        MAXI_rdata   : out std_logic_vector(C_MAXI_DWIDTH - 1 downto 0);
        MAXI_rresp   : out std_logic_vector(1 downto 0);
        MAXI_rvalid  : out std_logic;
        MAXI_rready  : in  std_logic;

        -- Control AXI port
        CAXI_clk     : in  std_logic;
        CAXI_rstn    : in  std_logic;
        CAXI_awaddr  : in  std_logic_vector(C_CAXI_AWIDTH - 1 downto 0);
        CAXI_awvalid : in  std_logic;
        CAXI_awready : out std_logic;
        CAXI_wdata   : in  std_logic_vector(C_CAXI_DWIDTH - 1 downto 0);
        CAXI_wstrb   : in  std_logic_vector((C_CAXI_DWIDTH / 8) - 1 downto 0);
        CAXI_wvalid  : in  std_logic;
        CAXI_wready  : out std_logic;
        CAXI_bresp   : out std_logic_vector(1 downto 0);
        CAXI_bvalid  : out std_logic;
        CAXI_bready  : in  std_logic;
        CAXI_araddr  : in  std_logic_vector(C_CAXI_AWIDTH - 1 downto 0);
        CAXI_arvalid : in  std_logic;
        CAXI_arready : out std_logic;
        CAXI_rdata   : out std_logic_vector(C_CAXI_DWIDTH - 1 downto 0);
        CAXI_rresp   : out std_logic_vector(1 downto 0);
        CAXI_rvalid  : out std_logic;
        CAXI_rready  : in  std_logic;

        -- Clock gating port
        clk_out      : out std_logic
    );
end entity vivado;

architecture imp of vivado is
    component BUFGCE
        port(
            I  : in  std_logic;
            CE : in  std_logic;
            O  : out std_logic
        );
    end component;

    constant C_MEM_ROWS     : natural                       := 2 ** (ARRAYS_LOG + ROWS_LOG + WORDSIZE_LOG - 3);
    constant C_MEM_MIN_SIZE : std_logic_vector(31 downto 0) := std_logic_vector(to_unsigned(C_MEM_ROWS - 1, 32));

    constant C_MEM_ADDR_RANGE_ARRAY : SLV64_ARRAY_TYPE(0 to 1) := (
        X"0000_0000_0000_0000",
        X"0000_0000" & C_MEM_MIN_SIZE
    );

    constant C_MEM_NUM_CE_ARRAY : INTEGER_ARRAY_TYPE(0 to 0) := (
        0 => 1
    );

    constant C_IPIC_NUM_REG : integer := 16;
    constant C_IPIC_NUM_MEM : integer := 2;

    constant C_IPIC_MIN_SIZE : std_logic_vector(31 downto 0) := X"000F_FFFF";

    constant C_IPIC_ADDR_RANGE_ARRAY : SLV64_ARRAY_TYPE(0 to 2 * (1 + C_IPIC_NUM_MEM) - 1) := (
        X"0000_0000_0000_0000",
        X"0000_0000_0000_FFFF",
        X"0000_0000_0001_0000",
        X"0000_0000_0001_FFFF",
        X"0000_0000_0002_0000",
        X"0000_0000_0002_FFFF"
    );

    constant C_IPIC_NUM_CE_ARRAY : INTEGER_ARRAY_TYPE(0 to (1 + C_IPIC_NUM_MEM) - 1) := (
        0 => C_IPIC_NUM_REG,
        1 => 1,
        2 => 1
    );

    signal clk, rst, clk_en, MAXI_rst : std_logic;

    signal mmem_rnw, cmem_rnw                             : std_logic;
    signal cmem_en, cmem_ack                              : std_logic;
    signal mmem_en, mmem_ack                              : std_logic_vector(0 downto 0);
    signal mmem_addr, cmem_addr                           : std_logic_vector(C_MAXI_AWIDTH - 1 downto 0);
    signal mmem_rdata, mmem_wdata, cmem_rdata, cmem_wdata : std_logic_vector(C_MAXI_DWIDTH - 1 downto 0);
    signal mmem_be, cmem_be                               : std_logic_vector(C_MAXI_DWIDTH / 8 - 1 downto 0);

    signal MIP2Bus_Data  : std_logic_vector(C_MAXI_DWIDTH - 1 downto 0);
    signal MIP2Bus_WrAck : std_logic;
    signal MIP2Bus_RdAck : std_logic;
    signal MIP2Bus_Error : std_logic;
    signal MBus2IP_Addr  : std_logic_vector(C_MAXI_AWIDTH - 1 downto 0);
    signal MBus2IP_Data  : std_logic_vector(C_MAXI_DWIDTH - 1 downto 0);
    signal MBus2IP_RNW   : std_logic;
    signal MBus2IP_BE    : std_logic_vector(C_MAXI_DWIDTH / 8 - 1 downto 0);
    signal MBus2IP_CS    : std_logic_vector(0 to 0);

    signal CIP2Bus_Data  : std_logic_vector(C_CAXI_DWIDTH - 1 downto 0);
    signal CIP2Bus_WrAck : std_logic;
    signal CIP2Bus_RdAck : std_logic;
    signal CIP2Bus_Error : std_logic;
    signal CBus2IP_Addr  : std_logic_vector(C_CAXI_AWIDTH - 1 downto 0);
    signal CBus2IP_Data  : std_logic_vector(C_CAXI_DWIDTH - 1 downto 0);
    signal CBus2IP_RNW   : std_logic;
    signal CBus2IP_BE    : std_logic_vector(C_CAXI_DWIDTH / 8 - 1 downto 0);
    signal CBus2IP_CS    : std_logic_vector(0 to (1 + C_IPIC_NUM_MEM) - 1);
    signal CBus2IP_RdCE  : std_logic_vector(0 to C_IPIC_NUM_REG + C_IPIC_NUM_MEM - 1);
    signal CBus2IP_WrCE  : std_logic_vector(0 to C_IPIC_NUM_REG + C_IPIC_NUM_MEM - 1);
begin
    clk      <= CAXI_clk;
    rst      <= not CAXI_rstn;
    MAXI_rst <= not MAXI_rstn;

    CLKGATE : BUFGCE
        port map(
            I  => clk,
            CE => clk_en,
            O  => clk_out
        );

    MAXI_IPIF : entity axi_lite_ipif_v3_0_4.axi_lite_ipif
        generic map(
            C_S_AXI_DATA_WIDTH     => C_MAXI_DWIDTH,
            C_S_AXI_ADDR_WIDTH     => C_MAXI_AWIDTH,
            C_S_AXI_MIN_SIZE       => C_MEM_MIN_SIZE,
            C_USE_WSTRB            => 1,
            C_DPHASE_TIMEOUT       => 0,
            C_ARD_ADDR_RANGE_ARRAY => C_MEM_ADDR_RANGE_ARRAY,
            C_ARD_NUM_CE_ARRAY     => C_MEM_NUM_CE_ARRAY,
            C_FAMILY               => C_FAMILY
        )
        port map(
            S_AXI_ACLK    => MAXI_clk,
            S_AXI_ARESETN => MAXI_rstn,
            S_AXI_AWADDR  => MAXI_awaddr,
            S_AXI_AWVALID => MAXI_awvalid,
            S_AXI_AWREADY => MAXI_awready,
            S_AXI_WDATA   => MAXI_wdata,
            S_AXI_WSTRB   => MAXI_wstrb,
            S_AXI_WVALID  => MAXI_wvalid,
            S_AXI_WREADY  => MAXI_wready,
            S_AXI_BRESP   => MAXI_bresp,
            S_AXI_BVALID  => MAXI_bvalid,
            S_AXI_BREADY  => MAXI_bready,
            S_AXI_ARADDR  => MAXI_araddr,
            S_AXI_ARVALID => MAXI_arvalid,
            S_AXI_ARREADY => MAXI_arready,
            S_AXI_RDATA   => MAXI_rdata,
            S_AXI_RRESP   => MAXI_rresp,
            S_AXI_RVALID  => MAXI_rvalid,
            S_AXI_RREADY  => MAXI_rready,
            Bus2IP_Clk    => open,
            Bus2IP_Resetn => open,
            Bus2IP_Addr   => MBus2IP_Addr,
            Bus2IP_RNW    => MBus2IP_RNW,
            Bus2IP_BE     => MBus2IP_BE,
            Bus2IP_CS     => MBus2IP_CS,
            Bus2IP_RdCE   => open,
            Bus2IP_WrCE   => open,
            Bus2IP_Data   => MBus2IP_Data,
            IP2Bus_Data   => MIP2Bus_Data,
            IP2Bus_WrAck  => MIP2Bus_WrAck,
            IP2Bus_RdAck  => MIP2Bus_RdAck,
            IP2Bus_Error  => MIP2Bus_Error
        );

    MAXI_IPIC_SLAVE : entity work.ipic_slave
        generic map(
            C_IPIC_AWIDTH  => C_MAXI_AWIDTH,
            C_IPIC_DWIDTH  => C_MAXI_DWIDTH,
            C_IPIC_NUM_REG => 0,
            C_IPIC_NUM_MEM => 1,
            DEBUG          => DEBUG
        )
        port map(
            clk          => MAXI_clk,
            rst          => MAXI_rst,
            Bus2IP_Addr  => MBus2IP_Addr,
            Bus2IP_CS    => MBus2IP_CS,
            Bus2IP_RNW   => MBus2IP_RNW,
            Bus2IP_Data  => MBus2IP_Data,
            Bus2IP_BE    => MBus2IP_BE,
            Bus2IP_RdCE  => "",
            Bus2IP_WrCE  => "",
            IP2Bus_Data  => MIP2Bus_Data,
            IP2Bus_RdAck => MIP2Bus_RdAck,
            IP2Bus_WrAck => MIP2Bus_WrAck,
            IP2Bus_Error => MIP2Bus_Error,
            regs_rdata   => "",
            regs_wen     => open,
            regs_wdata   => open,
            mem_en       => mmem_en,
            mem_rnw      => mmem_rnw,
            mem_addr     => mmem_addr,
            mem_wdata    => mmem_wdata,
            mem_be       => mmem_be,
            mem_rdata    => mmem_rdata,
            mem_ack      => mmem_ack
        );

    CROSSING : entity work.mem_crossing
        generic map(
            C_IPIC_AWIDTH => C_MAXI_AWIDTH,
            C_IPIC_DWIDTH => C_MAXI_DWIDTH
        )
        port map(
            clk      => clk,
            rst      => rst,
            s_clk_en => clk_en,
            s_en     => mmem_en(0),
            s_rnw    => mmem_rnw,
            s_addr   => mmem_addr,
            s_wdata  => mmem_wdata,
            s_be     => mmem_be,
            s_rdata  => mmem_rdata,
            s_ack    => mmem_ack(0),
            m_en     => cmem_en,
            m_rnw    => cmem_rnw,
            m_addr   => cmem_addr,
            m_wdata  => cmem_wdata,
            m_be     => cmem_be,
            m_rdata  => cmem_rdata,
            m_ack    => cmem_ack
        );

    CAXI_IPIF : entity axi_lite_ipif_v3_0_4.axi_lite_ipif
        generic map(
            C_S_AXI_DATA_WIDTH     => C_CAXI_DWIDTH,
            C_S_AXI_ADDR_WIDTH     => C_CAXI_AWIDTH,
            C_S_AXI_MIN_SIZE       => C_IPIC_MIN_SIZE,
            C_USE_WSTRB            => 1,
            C_DPHASE_TIMEOUT       => 0,
            C_ARD_ADDR_RANGE_ARRAY => C_IPIC_ADDR_RANGE_ARRAY,
            C_ARD_NUM_CE_ARRAY     => C_IPIC_NUM_CE_ARRAY,
            C_FAMILY               => C_FAMILY
        )
        port map(
            S_AXI_ACLK    => CAXI_clk,
            S_AXI_ARESETN => CAXI_rstn,
            S_AXI_AWADDR  => CAXI_awaddr,
            S_AXI_AWVALID => CAXI_awvalid,
            S_AXI_AWREADY => CAXI_awready,
            S_AXI_WDATA   => CAXI_wdata,
            S_AXI_WSTRB   => CAXI_wstrb,
            S_AXI_WVALID  => CAXI_wvalid,
            S_AXI_WREADY  => CAXI_wready,
            S_AXI_BRESP   => CAXI_bresp,
            S_AXI_BVALID  => CAXI_bvalid,
            S_AXI_BREADY  => CAXI_bready,
            S_AXI_ARADDR  => CAXI_araddr,
            S_AXI_ARVALID => CAXI_arvalid,
            S_AXI_ARREADY => CAXI_arready,
            S_AXI_RDATA   => CAXI_rdata,
            S_AXI_RRESP   => CAXI_rresp,
            S_AXI_RVALID  => CAXI_rvalid,
            S_AXI_RREADY  => CAXI_rready,
            Bus2IP_Clk    => open,
            Bus2IP_Resetn => open,
            Bus2IP_Addr   => CBus2IP_Addr,
            Bus2IP_RNW    => CBus2IP_RNW,
            Bus2IP_BE     => CBus2IP_BE,
            Bus2IP_CS     => CBus2IP_CS,
            Bus2IP_RdCE   => CBus2IP_RdCE,
            Bus2IP_WrCE   => CBus2IP_WrCE,
            Bus2IP_Data   => CBus2IP_Data,
            IP2Bus_Data   => CIP2Bus_Data,
            IP2Bus_WrAck  => CIP2Bus_WrAck,
            IP2Bus_RdAck  => CIP2Bus_RdAck,
            IP2Bus_Error  => CIP2Bus_Error
        );

    TOPLEVEL : entity work.toplevel
        generic map(
            C_MEM_AWIDTH      => C_MAXI_AWIDTH,
            C_MEM_DWIDTH_LOG  => C_MAXI_DWIDTH_LOG,
            C_IPIC_AWIDTH     => C_CAXI_AWIDTH,
            C_IPIC_DWIDTH_LOG => C_CAXI_DWIDTH_LOG,
            C_IPIC_NUM_REG    => C_IPIC_NUM_REG,
            C_IPIC_NUM_MEM    => C_IPIC_NUM_MEM,
            WORDSIZE_LOG      => WORDSIZE_LOG,
            ARRAYS_LOG        => ARRAYS_LOG,
            ROWS_LOG          => ROWS_LOG,
            TIMERSIZE         => TIMERSIZE,
            TDIFFSIZE         => TDIFFSIZE,
            ACTSIZE           => ACTSIZE,
            SKIPSIZE          => SKIPSIZE,
            READWRITE         => READWRITE,
            MEMARRAY_MODEL    => MEMARRAY_MODEL,
            MOD_MODEL         => MOD_MODEL,
            NEVENTS_LOG       => NEVENTS_LOG,
            PRAM_INITFILE     => PRAM_INITFILE,
            EVRAM_INITFILE    => EVRAM_INITFILE,
            DEBUG             => DEBUG
        )
        port map(
            clk          => clk,
            rst_ext      => rst,
            clk_en       => clk_en,
            mem_en       => cmem_en,
            mem_rnw      => cmem_rnw,
            mem_addr     => cmem_addr,
            mem_wdata    => cmem_wdata,
            mem_be       => cmem_be,
            mem_rdata    => cmem_rdata,
            mem_ack      => cmem_ack,
            Bus2IP_Addr  => CBus2IP_Addr,
            Bus2IP_Data  => CBus2IP_Data,
            Bus2IP_RNW   => CBus2IP_RNW,
            Bus2IP_BE    => CBus2IP_BE,
            Bus2IP_CS    => CBus2IP_CS,
            Bus2IP_RdCE  => CBus2IP_RdCE,
            Bus2IP_WrCE  => CBus2IP_WrCE,
            IP2Bus_Data  => CIP2Bus_Data,
            IP2Bus_WrAck => CIP2Bus_WrAck,
            IP2Bus_RdAck => CIP2Bus_RdAck,
            IP2Bus_Error => CIP2Bus_Error
        );

end architecture imp;
