library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity toplevel is
    generic(
        -- Memory port generics
        C_MEM_AWIDTH      : natural;
        C_MEM_DWIDTH_LOG  : natural;

        -- IPIC port generics
        C_IPIC_AWIDTH     : natural;
        C_IPIC_DWIDTH_LOG : natural;
        C_IPIC_NUM_REG    : natural;
        C_IPIC_NUM_MEM    : natural;

        -- User generics
        WORDSIZE_LOG      : natural;
        ARRAYS_LOG        : natural;
        ROWS_LOG          : natural;
        TIMERSIZE         : natural;
        TDIFFSIZE         : natural;
        ACTSIZE           : natural;
        SKIPSIZE          : natural;
        READWRITE         : boolean;
        MEMARRAY_MODEL    : string;
        MOD_MODEL         : string;
        NEVENTS_LOG       : natural;
        PRAM_INITFILE     : string  := "";
        EVRAM_INITFILE    : string  := "";
        DEBUG             : boolean := false
    );
    port(
        clk          : in  std_logic;
        rst_ext      : in  std_logic;

        -- Clock gating port
        clk_en       : out std_logic;

        -- Memory port
        mem_en       : in  std_logic;
        mem_rnw      : in  std_logic;
        mem_addr     : in  std_logic_vector(C_MEM_AWIDTH - 1 downto 0);
        mem_wdata    : in  std_logic_vector(2 ** C_MEM_DWIDTH_LOG - 1 downto 0);
        mem_be       : in  std_logic_vector(2 ** C_MEM_DWIDTH_LOG / 8 - 1 downto 0);
        mem_rdata    : out std_logic_vector(2 ** C_MEM_DWIDTH_LOG - 1 downto 0);
        mem_ack      : out std_logic;

        -- IPIC port
        Bus2IP_Addr  : in  std_logic_vector(C_IPIC_AWIDTH - 1 downto 0);
        Bus2IP_Data  : in  std_logic_vector(2 ** C_IPIC_DWIDTH_LOG - 1 downto 0);
        Bus2IP_RNW   : in  std_logic;
        Bus2IP_BE    : in  std_logic_vector(2 ** C_IPIC_DWIDTH_LOG / 8 - 1 downto 0);
        Bus2IP_CS    : in  std_logic_vector(0 to (1 + C_IPIC_NUM_MEM) - 1);
        Bus2IP_RdCE  : in  std_logic_vector(0 to C_IPIC_NUM_REG + C_IPIC_NUM_MEM - 1);
        Bus2IP_WrCE  : in  std_logic_vector(0 to C_IPIC_NUM_REG + C_IPIC_NUM_MEM - 1);
        IP2Bus_Data  : out std_logic_vector(2 ** C_IPIC_DWIDTH_LOG - 1 downto 0);
        IP2Bus_WrAck : out std_logic;
        IP2Bus_RdAck : out std_logic;
        IP2Bus_Error : out std_logic
    );
end toplevel;

architecture imp of toplevel is
    constant WORDSIZE      : natural := 2 ** WORDSIZE_LOG;
    constant ARRAYS        : natural := 2 ** ARRAYS_LOG;
    constant C_IPIC_DWIDTH : natural := 2 ** C_IPIC_DWIDTH_LOG;

    function count_bits(value : std_logic_vector) return unsigned is
        variable count : unsigned(C_IPIC_DWIDTH - 1 downto 0) := (others => '0');
    begin
        for i in value'range loop
            if value(i) = '1' then
                count := count + 1;
            end if;
        end loop;
        return count;
    end function;

    type regs_type is array (0 to C_IPIC_NUM_REG - 1) of std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);
    signal plb_regs : regs_type;

    signal plb_regs_data  : std_logic_vector(C_IPIC_NUM_REG * C_IPIC_DWIDTH - 1 downto 0);
    signal plb_regs_wen   : std_logic_vector(C_IPIC_NUM_REG - 1 downto 0);
    signal plb_regs_wdata : std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);

    signal rst : std_logic;

    signal simts, cputs       : std_logic_vector(TIMERSIZE - 1 downto 0);
    signal simts_en, cputs_en : std_logic;

    signal rfint                 : std_logic_vector(TIMERSIZE - 1 downto 0);
    signal ten, sen, gen, rfdist : std_logic;
    signal rfskip                : std_logic_vector(SKIPSIZE - 1 downto 0);

    signal read_strobe, write_strobe, stall_strobe                                   : std_logic;
    signal refresh_strobe, flip_strobe                                               : std_logic_vector(ARRAYS - 1 downto 0);
    signal read_counter, write_counter, refresh_counter, flip_counter, stall_counter : unsigned(C_IPIC_DWIDTH - 1 downto 0);
    signal mod_sel                                                                   : std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);

    signal stall_ren, stall_rack, stall_wen, stall_wack, stall_en : std_logic;
    signal stall_raddr, stall_waddr                               : std_logic_vector(ARRAYS_LOG + ROWS_LOG - 1 downto 0);
    signal stall_rdata, stall_wdata                               : std_logic_vector(WORDSIZE - 1 downto 0);

    signal mux_ren, mux_rack, mux_wen, mux_wack : std_logic;
    signal mux_raddr, mux_waddr                 : std_logic_vector(ARRAYS_LOG + ROWS_LOG - 1 downto 0);
    signal mux_rdata, mux_wdata                 : std_logic_vector(WORDSIZE - 1 downto 0);

    signal ctrl_ren, ctrl_rack, ctrl_wen, ctrl_wack : std_logic_vector(ARRAYS - 1 downto 0);
    signal ctrl_raddr, ctrl_waddr                   : std_logic_vector(ROWS_LOG - 1 downto 0);
    signal ctrl_rdata                               : std_logic_vector(ARRAYS * WORDSIZE - 1 downto 0);
    signal ctrl_wdata                               : std_logic_vector(WORDSIZE - 1 downto 0);

    signal sync_ready                                           : std_logic;
    signal sync_ren, sync_rack, sync_wen, sync_wack, sync_tstop : std_logic_vector(ARRAYS - 1 downto 0);
    signal sync_raddr, sync_waddr                               : std_logic_vector(ARRAYS * ROWS_LOG - 1 downto 0);
    signal sync_rdata, sync_wdata                               : std_logic_vector(ARRAYS * WORDSIZE - 1 downto 0);

    signal arr_ren, arr_rack, arr_wen, arr_wack : std_logic_vector(ARRAYS - 1 downto 0);
    signal arr_raddr, arr_waddr                 : std_logic_vector(ARRAYS * ROWS_LOG - 1 downto 0);
    signal arr_rdata, arr_wdata                 : std_logic_vector(ARRAYS * WORDSIZE - 1 downto 0);

    signal plb_mem_en    : std_logic_vector(C_IPIC_NUM_MEM - 1 downto 0);
    signal plb_mem_rnw   : std_logic;
    signal plb_mem_addr  : std_logic_vector(C_IPIC_AWIDTH - 1 downto 0);
    signal plb_mem_wdata : std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);
    signal plb_mem_be    : std_logic_vector(C_IPIC_DWIDTH / 8 - 1 downto 0);
    signal plb_mem_rdata : std_logic_vector(C_IPIC_NUM_MEM * C_IPIC_DWIDTH - 1 downto 0);
    signal plb_mem_ack   : std_logic_vector(C_IPIC_NUM_MEM - 1 downto 0);

    signal pram_wen, evram_wen : std_logic;
    signal pram_waddr          : std_logic_vector(ROWS_LOG - 1 downto 0);
    signal evram_waddr         : std_logic_vector(NEVENTS_LOG - 1 downto 0);
    signal pram_wdata          : std_logic_vector(2 * NEVENTS_LOG - 1 downto 0);
    signal evram_wdata         : std_logic_vector(ACTSIZE + TDIFFSIZE + WORDSIZE_LOG - 1 downto 0);

    signal mod_en, mod_ack, mod_pram_wen, mod_evram_wen : std_logic_vector(ARRAYS - 1 downto 0);
    signal mod_addr                                     : std_logic_vector(ARRAYS * ROWS_LOG - 1 downto 0);
    signal mod_tdiff                                    : std_logic_vector(ARRAYS * TDIFFSIZE - 1 downto 0);
    signal mod_data_in, mod_data_out                    : std_logic_vector(ARRAYS * WORDSIZE - 1 downto 0);
begin
    MEM_SLAVE : entity work.mem_slave
        generic map(
            AWIDTH       => C_MEM_AWIDTH,
            DWIDTH_LOG   => C_MEM_DWIDTH_LOG,
            WORDSIZE_LOG => WORDSIZE_LOG,
            ROWS_LOG     => ARRAYS_LOG + ROWS_LOG,
            DEBUG        => DEBUG
        )
        port map(
            clk          => clk,
            rst          => rst,
            en           => mem_en,
            rnw          => mem_rnw,
            addr         => mem_addr,
            wdata        => mem_wdata,
            be           => mem_be,
            rdata        => mem_rdata,
            ack          => mem_ack,
            m_raddr      => stall_raddr,
            m_ren        => stall_ren,
            m_rdata      => stall_rdata,
            m_rack       => stall_rack,
            m_waddr      => stall_waddr,
            m_wen        => stall_wen,
            m_wdata      => stall_wdata,
            m_wack       => stall_wack,
            read_strobe  => read_strobe,
            write_strobe => write_strobe
        );

    SIMTIMER : entity work.timer
        generic map(
            TIMERSIZE => TIMERSIZE
        )
        port map(
            clk => clk,
            rst => rst,
            en  => simts_en,
            ts  => simts
        );

    simts_en <= ten and sync_ready;
    clk_en   <= sync_ready or not gen;

    CPUTIMER : entity work.timer
        generic map(
            TIMERSIZE => TIMERSIZE
        )
        port map(
            clk => clk,
            rst => rst,
            en  => cputs_en,
            ts  => cputs
        );

    cputs_en <= ten and ((simts_en and stall_en) or (not stall_wen and not stall_ren));

    MEMSTALL : entity work.memstall
        generic map(
            WORDSIZE_LOG => WORDSIZE_LOG,
            ROWS_LOG     => ARRAYS_LOG + ROWS_LOG
        )
        port map(
            s_raddr => stall_raddr,
            s_ren   => stall_ren,
            s_rdata => stall_rdata,
            s_rack  => stall_rack,
            s_waddr => stall_waddr,
            s_wen   => stall_wen,
            s_wdata => stall_wdata,
            s_wack  => stall_wack,
            m_raddr => mux_raddr,
            m_ren   => mux_ren,
            m_rdata => mux_rdata,
            m_rack  => mux_rack,
            m_waddr => mux_waddr,
            m_wen   => mux_wen,
            m_wdata => mux_wdata,
            m_wack  => mux_wack,
            en      => stall_en
        );

    stall_en <= '1' when sen = '0' or ten = '0' or simts = cputs else '0';

    stall_strobe <= simts_en and (mux_ren or mux_wen);

    MUX : entity work.memmux
        generic map(
            CHANNELS_LOG => ARRAYS_LOG,
            WORDSIZE_LOG => WORDSIZE_LOG,
            S_ROWS_LOG   => ARRAYS_LOG + ROWS_LOG,
            M_ROWS_LOG   => ROWS_LOG
        )
        port map(
            clk     => clk,
            rst     => rst,
            s_raddr => mux_raddr,
            s_ren   => mux_ren,
            s_rdata => mux_rdata,
            s_rack  => mux_rack,
            s_waddr => mux_waddr,
            s_wen   => mux_wen,
            s_wdata => mux_wdata,
            s_wack  => mux_wack,
            m_raddr => ctrl_raddr,
            m_ren   => ctrl_ren,
            m_rdata => ctrl_rdata,
            m_rack  => ctrl_rack,
            m_waddr => ctrl_waddr,
            m_wen   => ctrl_wen,
            m_wdata => ctrl_wdata,
            m_wack  => ctrl_wack
        );

    CTRL_LOOP : for i in 0 to ARRAYS - 1 generate
        MEMCTRL : entity work.memctrl
            generic map(
                WORDSIZE_LOG => WORDSIZE_LOG,
                ROWS_LOG     => ROWS_LOG,
                TIMERSIZE    => TIMERSIZE,
                SKIPSIZE     => SKIPSIZE,
                READWRITE    => READWRITE,
                DEBUG        => DEBUG
            )
            port map(
                clk            => clk,
                rst            => rst,
                s_raddr        => ctrl_raddr,
                s_ren          => ctrl_ren(i),
                s_rdata        => ctrl_rdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                s_rack         => ctrl_rack(i),
                s_waddr        => ctrl_waddr,
                s_wen          => ctrl_wen(i),
                s_wdata        => ctrl_wdata,
                s_wack         => ctrl_wack(i),
                s_ready        => open,
                m_raddr        => sync_raddr((i + 1) * ROWS_LOG - 1 downto i * ROWS_LOG),
                m_ren          => sync_ren(i),
                m_rdata        => sync_rdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                m_rack         => sync_rack(i),
                m_waddr        => sync_waddr((i + 1) * ROWS_LOG - 1 downto i * ROWS_LOG),
                m_wen          => sync_wen(i),
                m_wdata        => sync_wdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                m_wack         => sync_wack(i),
                m_ready        => sync_ready,
                ts             => simts,
                rfint          => rfint,
                rfskip_log     => rfskip,
                rfdist         => rfdist,
                refresh_strobe => refresh_strobe(i),
                stop           => sync_tstop(i)
            );
    end generate;

    SYNC : entity work.synchronizer
        generic map(
            CHANNELS     => ARRAYS,
            WORDSIZE_LOG => WORDSIZE_LOG,
            ROWS_LOG     => ROWS_LOG
        )
        port map(
            clk     => clk,
            rst     => rst,
            s_raddr => sync_raddr,
            s_ren   => sync_ren,
            s_rdata => sync_rdata,
            s_rack  => sync_rack,
            s_waddr => sync_waddr,
            s_wen   => sync_wen,
            s_wdata => sync_wdata,
            s_wack  => sync_wack,
            s_tstop => sync_tstop,
            s_ready => sync_ready,
            m_raddr => arr_raddr,
            m_ren   => arr_ren,
            m_rdata => arr_rdata,
            m_rack  => arr_rack,
            m_waddr => arr_waddr,
            m_wen   => arr_wen,
            m_wdata => arr_wdata,
            m_wack  => arr_wack
        );

    ARR_LOOP : for i in 0 to ARRAYS - 1 generate
        assert MEMARRAY_MODEL = "IDEAL" or MEMARRAY_MODEL = "TS" report "Invalid memarray model" severity failure;

        MEMARRAY_BRAM : if MEMARRAY_MODEL = "IDEAL" generate
            MEMARRAY : entity work.memarray_bram
                generic map(
                    WORDSIZE_LOG => WORDSIZE_LOG,
                    ROWS_LOG     => ROWS_LOG,
                    DEBUG        => DEBUG
                )
                port map(
                    clk   => clk,
                    rst   => rst,
                    raddr => arr_raddr((i + 1) * ROWS_LOG - 1 downto i * ROWS_LOG),
                    ren   => arr_ren(i),
                    rdata => arr_rdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                    rack  => arr_rack(i),
                    waddr => arr_waddr((i + 1) * ROWS_LOG - 1 downto i * ROWS_LOG),
                    wen   => arr_wen(i),
                    wdata => arr_wdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                    wack  => arr_wack(i),
                    ready => open
                );
        end generate;

        MEMARRAY_TS : if MEMARRAY_MODEL = "TS" generate
            MEMARRAY : entity work.memarray_ts
                generic map(
                    WORDSIZE_LOG => WORDSIZE_LOG,
                    ROWS_LOG     => ROWS_LOG,
                    TIMERSIZE    => TIMERSIZE,
                    TDIFFSIZE    => TDIFFSIZE,
                    DEBUG        => DEBUG
                )
                port map(
                    clk          => clk,
                    rst          => rst,
                    raddr        => arr_raddr((i + 1) * ROWS_LOG - 1 downto i * ROWS_LOG),
                    ren          => arr_ren(i),
                    rdata        => arr_rdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                    rack         => arr_rack(i),
                    waddr        => arr_waddr((i + 1) * ROWS_LOG - 1 downto i * ROWS_LOG),
                    wen          => arr_wen(i),
                    wdata        => arr_wdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                    wack         => arr_wack(i),
                    ready        => open,
                    mod_en       => mod_en(i),
                    mod_addr     => mod_addr((i + 1) * ROWS_LOG - 1 downto i * ROWS_LOG),
                    mod_tdiff    => mod_tdiff((i + 1) * TDIFFSIZE - 1 downto i * TDIFFSIZE),
                    mod_data_in  => mod_data_in((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                    mod_data_out => mod_data_out((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                    mod_ack      => mod_ack(i),
                    ts           => simts
                );

            assert MOD_MODEL = "SIMPLE" or MOD_MODEL = "EV" report "Invalid modifier model" severity failure;

            MOD_SIMPLE : if MOD_MODEL = "SIMPLE" generate
                MODIFIER : entity work.mod_simple
                    generic map(
                        WORDSIZE_LOG => WORDSIZE_LOG,
                        ROWS_LOG     => ROWS_LOG,
                        TDIFFSIZE    => TDIFFSIZE,
                        DEBUG        => DEBUG
                    )
                    port map(
                        clk      => clk,
                        rst      => rst,
                        en       => mod_en(i),
                        addr     => mod_addr((i + 1) * ROWS_LOG - 1 downto i * ROWS_LOG),
                        tdiff    => mod_tdiff((i + 1) * TDIFFSIZE - 1 downto i * TDIFFSIZE),
                        data_in  => mod_data_in((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                        data_out => mod_data_out((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                        ack      => mod_ack(i),
                        ready    => open
                    );
            end generate;

            MOD_EV : if MOD_MODEL = "EV" generate
                mod_pram_wen(i)  <= pram_wen when mod_sel = (mod_sel'range => '0') or unsigned(mod_sel) = to_unsigned(i + 1, C_IPIC_DWIDTH) else '0';
                mod_evram_wen(i) <= evram_wen when mod_sel = (mod_sel'range => '0') or unsigned(mod_sel) = to_unsigned(i + 1, C_IPIC_DWIDTH) else '0';

                MODIFIER : entity work.mod_ev
                    generic map(
                        WORDSIZE_LOG   => WORDSIZE_LOG,
                        ROWS_LOG       => ROWS_LOG,
                        TDIFFSIZE      => TDIFFSIZE,
                        ACTSIZE        => ACTSIZE,
                        NEVENTS_LOG    => NEVENTS_LOG,
                        PRAM_INITFILE  => PRAM_INITFILE,
                        EVRAM_INITFILE => EVRAM_INITFILE,
                        DEBUG          => DEBUG
                    )
                    port map(
                        clk         => clk,
                        rst         => rst,
                        en          => mod_en(i),
                        addr        => mod_addr((i + 1) * ROWS_LOG - 1 downto i * ROWS_LOG),
                        tdiff       => mod_tdiff((i + 1) * TDIFFSIZE - 1 downto i * TDIFFSIZE),
                        data_in     => mod_data_in((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                        data_out    => mod_data_out((i + 1) * WORDSIZE - 1 downto i * WORDSIZE),
                        ack         => mod_ack(i),
                        ready       => open,
                        pram_wen    => mod_pram_wen(i),
                        pram_waddr  => pram_waddr,
                        pram_wdata  => pram_wdata,
                        evram_wen   => mod_evram_wen(i),
                        evram_waddr => evram_waddr,
                        evram_wdata => evram_wdata,
                        flip_strobe => flip_strobe(i)
                    );
            end generate;
        end generate;
    end generate;

    IPIC_SLAVE : entity work.ipic_slave
        generic map(
            C_IPIC_AWIDTH  => C_IPIC_AWIDTH,
            C_IPIC_DWIDTH  => C_IPIC_DWIDTH,
            C_IPIC_NUM_REG => C_IPIC_NUM_REG,
            C_IPIC_NUM_MEM => C_IPIC_NUM_MEM,
            DEBUG          => DEBUG
        )
        port map(
            clk          => clk,
            rst          => rst,
            Bus2IP_Addr  => Bus2IP_Addr,
            Bus2IP_CS    => Bus2IP_CS(1 to (1 + C_IPIC_NUM_MEM) - 1),
            Bus2IP_RNW   => Bus2IP_RNW,
            Bus2IP_Data  => Bus2IP_Data,
            Bus2IP_BE    => Bus2IP_BE,
            Bus2IP_RdCE  => Bus2IP_RdCE(0 to C_IPIC_NUM_REG - 1),
            Bus2IP_WrCE  => Bus2IP_WrCE(0 to C_IPIC_NUM_REG - 1),
            IP2Bus_Data  => IP2Bus_Data,
            IP2Bus_RdAck => IP2Bus_RdAck,
            IP2Bus_WrAck => IP2Bus_WrAck,
            IP2Bus_Error => IP2Bus_Error,
            regs_rdata   => plb_regs_data,
            regs_wen     => plb_regs_wen,
            regs_wdata   => plb_regs_wdata,
            mem_en       => plb_mem_en,
            mem_rnw      => plb_mem_rnw,
            mem_addr     => plb_mem_addr,
            mem_wdata    => plb_mem_wdata,
            mem_be       => plb_mem_be,
            mem_rdata    => plb_mem_rdata,
            mem_ack      => plb_mem_ack
        );

    PRAM_SLAVE : entity work.mem_slave_simple
        generic map(
            AWIDTH     => C_IPIC_AWIDTH,
            DWIDTH_LOG => C_IPIC_DWIDTH_LOG,
            MEM_AWIDTH => ROWS_LOG,
            MEM_DWIDTH => 2 * NEVENTS_LOG
        )
        port map(
            clk     => clk,
            rst     => rst,
            en      => plb_mem_en(0),
            rnw     => plb_mem_rnw,
            addr    => plb_mem_addr,
            wdata   => plb_mem_wdata,
            be      => plb_mem_be,
            rdata   => plb_mem_rdata(C_IPIC_DWIDTH - 1 downto 0),
            ack     => plb_mem_ack(0),
            m_raddr => open,
            m_ren   => open,
            m_rdata => (pram_wdata'range => 'X'),
            m_waddr => pram_waddr,
            m_wen   => pram_wen,
            m_wdata => pram_wdata
        );

    EVRAM_SLAVE : entity work.mem_slave_simple
        generic map(
            AWIDTH     => C_IPIC_AWIDTH,
            DWIDTH_LOG => C_IPIC_DWIDTH_LOG,
            MEM_AWIDTH => NEVENTS_LOG,
            MEM_DWIDTH => ACTSIZE + TDIFFSIZE + WORDSIZE_LOG
        )
        port map(
            clk     => clk,
            rst     => rst,
            en      => plb_mem_en(1),
            rnw     => plb_mem_rnw,
            addr    => plb_mem_addr,
            wdata   => plb_mem_wdata,
            be      => plb_mem_be,
            rdata   => plb_mem_rdata(2 * C_IPIC_DWIDTH - 1 downto C_IPIC_DWIDTH),
            ack     => plb_mem_ack(1),
            m_raddr => open,
            m_ren   => open,
            m_rdata => (evram_wdata'range => 'X'),
            m_waddr => evram_waddr,
            m_wen   => evram_wen,
            m_wdata => evram_wdata
        );

    process(clk, rst) is
    begin
        if rst = '1' then
            read_counter    <= (others => '0');
            write_counter   <= (others => '0');
            refresh_counter <= (others => '0');
            flip_counter    <= (others => '0');
            stall_counter   <= (others => '0');
        elsif rising_edge(clk) then
            read_counter    <= read_counter + count_bits(read_strobe & "");
            write_counter   <= write_counter + count_bits(write_strobe & "");
            refresh_counter <= refresh_counter + count_bits(refresh_strobe(0 downto 0));
            flip_counter    <= flip_counter + count_bits(flip_strobe(0 downto 0));
            stall_counter   <= stall_counter + count_bits(stall_strobe & "");
        end if;
    end process;

    process(clk, rst_ext) is
    begin
        if rst_ext = '1' then
            rst <= '1';
        elsif rising_edge(clk) then
            rst <= plb_regs_wen(0) and plb_regs_wdata(0);
        end if;
    end process;

    process(clk, rst) is
    begin
        if rst = '1' then
            ten     <= '0';
            sen     <= '0';
            gen     <= '0';
            rfdist  <= '0';
            rfint   <= (others => '0');
            rfskip  <= (others => '0');
            mod_sel <= (others => '0');
        elsif rising_edge(clk) then
            if plb_regs_wen(0) = '1' then
                ten    <= plb_regs_wdata(1);
                sen    <= plb_regs_wdata(2);
                gen    <= plb_regs_wdata(3);
                rfdist <= plb_regs_wdata(4);
            end if;
            if plb_regs_wen(1) = '1' then
                rfint <= plb_regs_wdata(rfint'range);
            end if;
            if plb_regs_wen(2) = '1' then
                rfskip <= plb_regs_wdata(rfskip'range);
            end if;
            if plb_regs_wen(9) = '1' then
                mod_sel <= plb_regs_wdata;
            end if;
        end if;
    end process;

    plb_regs(0) <= (C_IPIC_DWIDTH - 1 downto 5 => '0') & rfdist & gen & sen & ten & '0';
    plb_regs(1) <= (C_IPIC_DWIDTH - 1 downto rfint'length => '0') & rfint;
    plb_regs(2) <= (C_IPIC_DWIDTH - 1 downto rfskip'length => '0') & rfskip;
    plb_regs(3) <= (C_IPIC_DWIDTH - 1 downto simts'length => '0') & simts;
    plb_regs(4) <= std_logic_vector(read_counter);
    plb_regs(5) <= std_logic_vector(write_counter);
    plb_regs(6) <= std_logic_vector(refresh_counter);
    plb_regs(7) <= std_logic_vector(flip_counter);
    plb_regs(8) <= std_logic_vector(stall_counter);
    plb_regs(9) <= mod_sel;

    PLB_REGS_GEN : for i in 0 to C_IPIC_NUM_REG - 1 generate
        plb_regs_data((i + 1) * C_IPIC_DWIDTH - 1 downto i * C_IPIC_DWIDTH) <= plb_regs(i);
    end generate;

end architecture imp;
