library ieee;
use ieee.std_logic_1164.all;

entity synchronizer is
    generic(
        CHANNELS     : natural;
        WORDSIZE_LOG : natural;
        ROWS_LOG     : natural
    );
    port(
        clk     : in  std_logic;
        rst     : in  std_logic;

        -- Slave port
        s_raddr : in  std_logic_vector(CHANNELS * ROWS_LOG - 1 downto 0);
        s_ren   : in  std_logic_vector(CHANNELS - 1 downto 0);
        s_rdata : out std_logic_vector(CHANNELS * (2 ** WORDSIZE_LOG) - 1 downto 0);
        s_rack  : out std_logic_vector(CHANNELS - 1 downto 0);
        s_waddr : in  std_logic_vector(CHANNELS * ROWS_LOG - 1 downto 0);
        s_wen   : in  std_logic_vector(CHANNELS - 1 downto 0);
        s_wdata : in  std_logic_vector(CHANNELS * (2 ** WORDSIZE_LOG) - 1 downto 0);
        s_wack  : out std_logic_vector(CHANNELS - 1 downto 0);
        s_tstop : in  std_logic_vector(CHANNELS - 1 downto 0);
        s_ready : out std_logic;

        -- Master port
        m_raddr : out std_logic_vector(CHANNELS * ROWS_LOG - 1 downto 0);
        m_ren   : out std_logic_vector(CHANNELS - 1 downto 0);
        m_rdata : in  std_logic_vector(CHANNELS * (2 ** WORDSIZE_LOG) - 1 downto 0);
        m_rack  : in  std_logic_vector(CHANNELS - 1 downto 0);
        m_waddr : out std_logic_vector(CHANNELS * ROWS_LOG - 1 downto 0);
        m_wen   : out std_logic_vector(CHANNELS - 1 downto 0);
        m_wdata : out std_logic_vector(CHANNELS * (2 ** WORDSIZE_LOG) - 1 downto 0);
        m_wack  : in  std_logic_vector(CHANNELS - 1 downto 0)
    );
end entity synchronizer;

architecture rtl of synchronizer is
    constant WORDSIZE : natural := 2 ** WORDSIZE_LOG;

    signal tstop_q                                  : std_logic_vector(CHANNELS - 1 downto 0);
    signal rack_reg, rack_next, wack_reg, wack_next : std_logic_vector(CHANNELS - 1 downto 0);
    signal ren_reg, ren_next, wen_reg, wen_next     : std_logic_vector(CHANNELS - 1 downto 0);
    signal rdata_reg, rdata_next                    : std_logic_vector(CHANNELS * (2 ** WORDSIZE_LOG) - 1 downto 0);
begin
    process(clk, rst) is
    begin
        if rst = '1' then
            tstop_q   <= (others => '0');
            rack_reg  <= (others => '0');
            wack_reg  <= (others => '0');
            ren_reg   <= (others => '0');
            wen_reg   <= (others => '0');
            rdata_reg <= (others => 'X');
        elsif rising_edge(clk) then
            tstop_q   <= s_tstop;
            rack_reg  <= rack_next;
            wack_reg  <= wack_next;
            ren_reg   <= ren_next;
            wen_reg   <= wen_next;
            rdata_reg <= rdata_next;
        end if;
    end process;

    process(tstop_q, rack_reg, wack_reg, ren_reg, wen_reg, rdata_reg, s_ren, s_wen, m_rack, m_rdata, m_wack) is
        variable ready : boolean;
    begin
        rack_next  <= rack_reg;
        wack_next  <= wack_reg;
        rdata_next <= rdata_reg;

        ren_next <= ren_reg and not m_rack;
        wen_next <= wen_reg and not m_wack;

        s_rack  <= (others => '0');
        s_wack  <= (others => '0');
        s_ready <= '0';
        s_rdata <= rdata_reg;

        ready := (tstop_q = (tstop_q'range => '0') and (ren_reg and not m_rack) = (ren_reg'range => '0') and (wen_reg and not m_wack) = (wen_reg'range => '0'));

        for i in 0 to CHANNELS - 1 loop
            if m_rack(i) = '1' then
                s_rdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE)    <= m_rdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE);
                rdata_next((i + 1) * WORDSIZE - 1 downto i * WORDSIZE) <= m_rdata((i + 1) * WORDSIZE - 1 downto i * WORDSIZE);
            end if;

            if ready or (tstop_q(i) = '1' and (ren_reg(i) = '0' or m_rack(i) = '1') and (wen_reg(i) = '0' or m_wack(i) = '1')) then
                s_rack(i) <= rack_reg(i);
                s_wack(i) <= wack_reg(i);

                rack_next(i) <= s_ren(i);
                ren_next(i)  <= s_ren(i);
                wack_next(i) <= s_wen(i);
                wen_next(i)  <= s_wen(i);

                rdata_next((i + 1) * WORDSIZE - 1 downto i * WORDSIZE) <= (others => 'X');
            end if;
        end loop;

        if ready then
            s_ready <= '1';
        else
            s_ready <= '0';
        end if;
    end process;

    m_ren   <= ren_next;
    m_wen   <= wen_next;
    m_raddr <= s_raddr;
    m_waddr <= s_waddr;
    m_wdata <= s_wdata;

end architecture rtl;
