library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_textio.all;
use std.textio.all;

entity mod_ev is
    generic(
        WORDSIZE_LOG   : natural;
        ROWS_LOG       : natural;
        TDIFFSIZE      : natural;
        ACTSIZE        : natural;
        NEVENTS_LOG    : natural;
        PRAM_INITFILE  : string  := "";
        EVRAM_INITFILE : string  := "";
        DEBUG          : boolean := false
    );
    port(
        clk         : in  std_logic;
        rst         : in  std_logic;

        -- Modifier port
        en          : in  std_logic;
        addr        : in  std_logic_vector(ROWS_LOG - 1 downto 0);
        tdiff       : in  std_logic_vector(TDIFFSIZE - 1 downto 0);
        data_in     : in  std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        data_out    : out std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        ack         : out std_logic;
        ready       : out std_logic;

        -- Pointer ram write port
        pram_wen    : in  std_logic;
        pram_waddr  : in  std_logic_vector(ROWS_LOG - 1 downto 0);
        pram_wdata  : in  std_logic_vector(2 * NEVENTS_LOG - 1 downto 0);

        -- Event ram write port
        evram_wen   : in  std_logic;
        evram_waddr : in  std_logic_vector(NEVENTS_LOG - 1 downto 0);
        evram_wdata : in  std_logic_vector(TDIFFSIZE + ACTSIZE + WORDSIZE_LOG - 1 downto 0);

        -- Control output port
        flip_strobe : out std_logic
    );
end entity mod_ev;

architecture rtl of mod_ev is
    constant WORDSIZE : natural := 2 ** WORDSIZE_LOG;
    type state_type is (IDLE, PTR, EV);
    type act_type is (NONE, DRT, DRT0, DRT1);

    signal pram_ren         : std_logic;
    signal pram_raddr       : std_logic_vector(ROWS_LOG - 1 downto 0);
    signal pram_rdata       : std_logic_vector(2 * NEVENTS_LOG - 1 downto 0);
    signal pram_ptr, pram_n : unsigned(NEVENTS_LOG - 1 downto 0);

    signal evram_ren   : std_logic;
    signal evram_raddr : std_logic_vector(NEVENTS_LOG - 1 downto 0);
    signal evram_rdata : std_logic_vector(TDIFFSIZE + ACTSIZE + WORDSIZE_LOG - 1 downto 0);
    signal evram_tdiff : std_logic_vector(TDIFFSIZE - 1 downto 0);
    signal evram_act   : unsigned(ACTSIZE - 1 downto 0);
    signal evram_bit   : unsigned(WORDSIZE_LOG - 1 downto 0);

    signal state_reg, state_next            : state_type;
    signal ptr_reg, ptr_next, n_reg, n_next : unsigned(NEVENTS_LOG - 1 downto 0);
    signal data_reg, data_next              : std_logic_vector(WORDSIZE - 1 downto 0);
    signal odata_reg, odata_next            : std_logic_vector(WORDSIZE - 1 downto 0);
    signal tdiff_reg, tdiff_next            : std_logic_vector(TDIFFSIZE - 1 downto 0);

    procedure apply_event(
        signal oinput : in std_logic_vector(WORDSIZE - 1 downto 0);
        signal input  : in std_logic_vector(WORDSIZE - 1 downto 0);
        signal bit    : in unsigned(WORDSIZE_LOG - 1 downto 0);
        signal act    : in unsigned(ACTSIZE - 1 downto 0);
        signal output : out std_logic_vector(WORDSIZE - 1 downto 0);
        signal strobe : out std_logic
    ) is
    begin
        output <= input;
        strobe <= '0';
        if ((to_integer(act) = act_type'pos(DRT)) or
            (to_integer(act) = act_type'pos(DRT0) and oinput(to_integer(bit)) = '0') or
            (to_integer(act) = act_type'pos(DRT1) and oinput(to_integer(bit)) = '1')
        ) then
            output(to_integer(bit)) <= not oinput(to_integer(bit));
            strobe                  <= '1';
        end if;
    end procedure;
begin
    PRAM : entity work.bram
        generic map(
            DWIDTH   => 2 * NEVENTS_LOG,
            AWIDTH   => ROWS_LOG,
            INITFILE => PRAM_INITFILE
        )
        port map(
            clk   => clk,
            rst   => rst,
            raddr => pram_raddr,
            ren   => pram_ren,
            rdata => pram_rdata,
            waddr => pram_waddr,
            wen   => pram_wen,
            wdata => pram_wdata
        );

    EVRAM : entity work.bram
        generic map(
            DWIDTH   => TDIFFSIZE + ACTSIZE + WORDSIZE_LOG,
            AWIDTH   => NEVENTS_LOG,
            INITFILE => EVRAM_INITFILE
        )
        port map(
            clk   => clk,
            rst   => rst,
            raddr => evram_raddr,
            ren   => evram_ren,
            rdata => evram_rdata,
            waddr => evram_waddr,
            wen   => evram_wen,
            wdata => evram_wdata
        );

    process(clk, rst) is
        variable debug_buf : line;
    begin
        if rst = '1' then
            state_reg <= IDLE;
            ptr_reg   <= (others => 'X');
            n_reg     <= (others => 'X');
            data_reg  <= (others => 'X');
            odata_reg <= (others => 'X');
            tdiff_reg <= (others => 'X');
        elsif rising_edge(clk) then
            if DEBUG and state_next = PTR then
                write(debug_buf, string'("[MOD ] modify 0x"));
                hwrite(debug_buf, data_in);
                write(debug_buf, string'(" addr=0x"));
                hwrite(debug_buf, addr);
                write(debug_buf, string'(" tdiff="));
                write(debug_buf, to_integer(unsigned(tdiff)));
                writeline(output, debug_buf);
            end if;

            state_reg <= state_next;
            ptr_reg   <= ptr_next;
            n_reg     <= n_next;
            data_reg  <= data_next;
            odata_reg <= odata_next;
            tdiff_reg <= tdiff_next;
        end if;
    end process;

    process(state_reg, ptr_reg, n_reg, data_reg, odata_reg, tdiff_reg, en, addr, tdiff, data_in, pram_ptr, pram_n, evram_tdiff, evram_act, evram_bit) is
        variable done : std_logic;
    begin
        done := '0';

        state_next <= state_reg;
        ptr_next   <= ptr_reg;
        n_next     <= n_reg;
        data_next  <= data_reg;
        odata_next <= odata_reg;
        tdiff_next <= tdiff_reg;

        data_out <= (others => 'X');
        ack      <= '0';
        ready    <= '0';

        pram_ren    <= '0';
        pram_raddr  <= (others => 'X');
        evram_ren   <= '0';
        evram_raddr <= (others => 'X');

        flip_strobe <= '0';

        case state_reg is
            when IDLE =>
                -- see below

            when PTR =>
                if not is_X(std_logic_vector(pram_n)) and pram_n /= (pram_n'range => '0') then
                    evram_ren   <= '1';
                    evram_raddr <= std_logic_vector(pram_ptr);
                    data_next   <= odata_reg;
                    ptr_next    <= pram_ptr + 1;
                    n_next      <= pram_n - 1;
                    state_next  <= EV;
                else
                    data_out <= odata_reg;

                    odata_next <= (others => 'X');
                    tdiff_next <= (others => 'X');
                    done       := '1';
                    state_next <= IDLE;
                end if;

            when EV =>
                if n_reg /= (n_reg'range => '0') and evram_tdiff <= tdiff_reg then
                    if not is_X(std_logic_vector(evram_bit)) then
                        apply_event(odata_reg, data_reg, evram_bit, evram_act, data_next, flip_strobe);
                    end if;

                    evram_ren   <= '1';
                    evram_raddr <= std_logic_vector(ptr_reg);
                    ptr_next    <= ptr_reg + 1;
                    n_next      <= n_reg - 1;
                else
                    data_out <= data_reg;
                    if evram_tdiff <= tdiff_reg and not is_X(std_logic_vector(evram_bit)) then
                        apply_event(odata_reg, data_reg, evram_bit, evram_act, data_out, flip_strobe);
                    end if;

                    data_next  <= (others => 'X');
                    odata_next <= (others => 'X');
                    tdiff_next <= (others => 'X');
                    ptr_next   <= (others => 'X');
                    n_next     <= (others => 'X');
                    done       := '1';
                    state_next <= IDLE;
                end if;

        end case;

        ack <= done;

        if state_reg = IDLE or done = '1' then
            ready <= '1';

            pram_ren   <= en;
            pram_raddr <= addr;

            if en = '1' then
                odata_next <= data_in;
                tdiff_next <= tdiff;
                state_next <= PTR;
            end if;
        end if;

    end process;

    pram_ptr <= unsigned(pram_rdata(2 * NEVENTS_LOG - 1 downto NEVENTS_LOG));
    pram_n   <= unsigned(pram_rdata(NEVENTS_LOG - 1 downto 0));

    evram_tdiff <= evram_rdata(TDIFFSIZE + ACTSIZE + WORDSIZE_LOG - 1 downto ACTSIZE + WORDSIZE_LOG);
    evram_act   <= unsigned(evram_rdata(ACTSIZE + WORDSIZE_LOG - 1 downto WORDSIZE_LOG));
    evram_bit   <= unsigned(evram_rdata(WORDSIZE_LOG - 1 downto 0));

end architecture rtl;
