library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity memmux is
    generic(
        CHANNELS_LOG : natural;
        WORDSIZE_LOG : natural;
        S_ROWS_LOG   : natural;
        M_ROWS_LOG   : natural
    );
    port(
        clk     : in  std_logic;
        rst     : in  std_logic;

        -- Slave port
        s_raddr : in  std_logic_vector(S_ROWS_LOG - 1 downto 0);
        s_ren   : in  std_logic;
        s_rdata : out std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        s_rack  : out std_logic;
        s_waddr : in  std_logic_vector(S_ROWS_LOG - 1 downto 0);
        s_wen   : in  std_logic;
        s_wdata : in  std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        s_wack  : out std_logic;

        -- Master port
        m_raddr : out std_logic_vector(M_ROWS_LOG - 1 downto 0);
        m_ren   : out std_logic_vector((2 ** CHANNELS_LOG) - 1 downto 0);
        m_rdata : in  std_logic_vector((2 ** CHANNELS_LOG) * (2 ** WORDSIZE_LOG) - 1 downto 0);
        m_rack  : in  std_logic_vector((2 ** CHANNELS_LOG) - 1 downto 0);
        m_waddr : out std_logic_vector(M_ROWS_LOG - 1 downto 0);
        m_wen   : out std_logic_vector((2 ** CHANNELS_LOG) - 1 downto 0);
        m_wdata : out std_logic_vector((2 ** WORDSIZE_LOG) - 1 downto 0);
        m_wack  : in  std_logic_vector((2 ** CHANNELS_LOG) - 1 downto 0)
    );
end entity memmux;

architecture rtl of memmux is
    constant WORDSIZE : natural := 2 ** WORDSIZE_LOG;

    signal rchannel, rchannel_q, wchannel, wchannel_q : unsigned(CHANNELS_LOG downto 0);
begin
    rchannel <= unsigned("0" & s_raddr(S_ROWS_LOG - 1 downto S_ROWS_LOG - CHANNELS_LOG));
    wchannel <= unsigned("0" & s_waddr(S_ROWS_LOG - 1 downto S_ROWS_LOG - CHANNELS_LOG));

    process(clk, rst) is
    begin
        if rst = '1' then
            rchannel_q <= (others => '0');
            wchannel_q <= (others => '0');
        elsif rising_edge(clk) then
            rchannel_q <= rchannel;
            wchannel_q <= wchannel;
        end if;
    end process;

    process(rchannel, wchannel, s_ren, s_wen) is
    begin
        m_ren                       <= (others => '0');
        m_ren(to_integer(rchannel)) <= s_ren;
        m_wen                       <= (others => '0');
        m_wen(to_integer(wchannel)) <= s_wen;
    end process;

    s_rack  <= m_rack(to_integer(rchannel_q));
    s_wack  <= m_wack(to_integer(wchannel_q));
    s_rdata <= m_rdata((to_integer(rchannel_q) + 1) * WORDSIZE - 1 downto to_integer(rchannel_q) * WORDSIZE);

    m_raddr <= s_raddr(M_ROWS_LOG - 1 downto 0);
    m_waddr <= s_waddr(M_ROWS_LOG - 1 downto 0);
    m_wdata <= s_wdata;

end architecture rtl;
