library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use std.textio.all;

entity memctrl is
    generic(
        WORDSIZE_LOG : natural;
        ROWS_LOG     : natural;
        TIMERSIZE    : natural;
        SKIPSIZE     : natural;
        READWRITE    : boolean;
        DEBUG        : boolean := false
    );
    port(
        clk            : in  std_logic;
        rst            : in  std_logic;

        -- Slave port
        s_raddr        : in  std_logic_vector(ROWS_LOG - 1 downto 0);
        s_ren          : in  std_logic;
        s_rdata        : out std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        s_rack         : out std_logic;
        s_waddr        : in  std_logic_vector(ROWS_LOG - 1 downto 0);
        s_wen          : in  std_logic;
        s_wdata        : in  std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        s_wack         : out std_logic;
        s_ready        : out std_logic;

        -- Master port
        m_raddr        : out std_logic_vector(ROWS_LOG - 1 downto 0);
        m_ren          : out std_logic;
        m_rdata        : in  std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        m_rack         : in  std_logic;
        m_waddr        : out std_logic_vector(ROWS_LOG - 1 downto 0);
        m_wen          : out std_logic;
        m_wdata        : out std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        m_wack         : in  std_logic;
        m_ready        : in  std_logic;

        -- Control input port
        ts             : in  std_logic_vector(TIMERSIZE - 1 downto 0);
        rfint          : in  std_logic_vector(TIMERSIZE - 1 downto 0);
        rfskip_log     : in  std_logic_vector(SKIPSIZE - 1 downto 0);
        rfdist         : in  std_logic;

        -- Control output port
        refresh_strobe : out std_logic;
        stop           : out std_logic
    );
end entity memctrl;

architecture rtl of memctrl is
    constant WORDSIZE : natural := 2 ** WORDSIZE_LOG;
    type state_type is (NORMAL, REFRESH);

    signal state_reg, state_next     : state_type                               := NORMAL;
    signal rfts_reg, rfts_next       : std_logic_vector(TIMERSIZE - 1 downto 0) := (others => '0');
    signal counter_reg, counter_next : unsigned(ROWS_LOG + 1 - 1 downto 0)      := (others => '0');
    signal data_reg, data_next       : std_logic_vector(WORDSIZE - 1 downto 0);
    signal first_reg, first_next     : std_logic;
    signal ren, ren_q                : std_logic;
    signal wen, wen_q                : std_logic;

    function do_stop(counter : unsigned(ROWS_LOG downto 0); first : std_logic; skip_log : std_logic_vector(SKIPSIZE - 1 downto 0); dist : std_logic) return std_logic is
        variable result : std_logic;
    begin
        result := '1';
        if unsigned(skip_log) = 0 then
            result := '0';
        elsif counter(ROWS_LOG) = '1' then
            result := '0';
        else
            for i in 1 to 2 ** SKIPSIZE - 1 loop
                if to_integer(unsigned(skip_log)) = i then
                    if dist = '1' and first = '0' and counter(i - 1 downto 0) = (i - 1 downto 0 => '0') then
                        result := '0';
                    end if;
                    if counter(i - 1 downto 0) = (i - 1 downto 0 => '1') then
                        result := '0';
                    end if;
                end if;
            end loop;
        end if;
        return result;
    end function;

    function is_last(counter : unsigned(ROWS_LOG downto 0); first : std_logic; skip_log : std_logic_vector(SKIPSIZE - 1 downto 0); dist : std_logic) return std_logic is
        variable result : std_logic;
    begin
        result := '0';
        if dist = '1' then
            if first = '0' then
                if unsigned(skip_log) = 0 then
                    result := '1';
                else
                    for i in 1 to 2 ** SKIPSIZE - 1 loop
                        if to_integer(unsigned(skip_log)) = i then
                            if counter(i - 1 downto 0) = (i - 1 downto 0 => '0') then
                                result := '1';
                            end if;
                        end if;
                    end loop;
                end if;
            end if;
        else
            result := counter(ROWS_LOG);
        end if;
        return result;
    end function;

    function counter2addr(counter : unsigned(ROWS_LOG downto 0); skip_log : std_logic_vector(SKIPSIZE - 1 downto 0)) return std_logic_vector is
        variable result : unsigned(ROWS_LOG - 1 downto 0);
    begin
        result := (others => 'X');
        if unsigned(skip_log) = 0 then
            result := counter(ROWS_LOG - 1 downto 0);
        else
            for i in 1 to 2 ** SKIPSIZE - 1 loop
                if to_integer(unsigned(skip_log)) = i then
                    result(ROWS_LOG - 1 downto ROWS_LOG - i) := counter(i - 1 downto 0);
                    result(ROWS_LOG - i - 1 downto 0)        := counter(ROWS_LOG - 1 downto i);
                end if;
            end loop;
        end if;
        return std_logic_vector(result);
    end function;
begin
    process(clk, rst) is
        variable debug_buf : line;
    begin
        if rst = '1' then
            state_reg   <= NORMAL;
            rfts_reg    <= (others => '0');
            counter_reg <= (others => '0');
            data_reg    <= (others => 'X');
            first_reg   <= '0';
            ren_q       <= '0';
            wen_q       <= '0';
        elsif rising_edge(clk) then
            if DEBUG and state_reg = NORMAL and state_next = REFRESH then
                write(debug_buf, string'("[MCTL] start refresh"));
                writeline(output, debug_buf);
            end if;
            if DEBUG and state_reg = REFRESH and state_next = NORMAL then
                write(debug_buf, string'("[MCTL] end refresh"));
                writeline(output, debug_buf);
            end if;

            state_reg   <= state_next;
            rfts_reg    <= rfts_next;
            counter_reg <= counter_next;
            data_reg    <= data_next;
            first_reg   <= first_next;
            ren_q       <= ren;
            wen_q       <= wen;
        end if;
    end process;

    process(state_reg, rfts_reg, counter_reg, data_reg, first_reg, ren_q, wen_q, s_raddr, s_ren, s_waddr, s_wen, s_wdata, m_rdata, m_rack, m_wack, m_ready, ts, rfint, rfskip_log, rfdist) is
        variable transparent : std_logic;
    begin
        transparent := '0';

        state_next   <= state_reg;
        rfts_next    <= rfts_reg;
        counter_next <= counter_reg;
        data_next    <= data_reg;
        first_next   <= first_reg;
        ren          <= ren_q;
        wen          <= wen_q;

        s_rdata <= (others => 'X');
        s_rack  <= '0';
        s_wack  <= '0';
        s_ready <= '0';

        m_raddr <= (others => 'X');
        m_waddr <= (others => 'X');
        m_wdata <= (others => 'X');

        stop           <= '0';
        refresh_strobe <= '0';

        case state_reg is
            when NORMAL =>
                s_rdata <= m_rdata;
                s_rack  <= m_rack;
                s_wack  <= m_wack;

                if m_rack = '1' then
                    ren <= '0';
                end if;
                if m_wack = '1' then
                    wen <= '0';
                end if;

                if unsigned(rfint) /= 0 and not is_X(ts) and unsigned(ts) >= unsigned(rfts_reg) + unsigned(rfint) then
                    if (ren_q = '0' or m_rack = '1') and (wen_q = '0' or m_wack = '1') then
                        if READWRITE then
                            ren     <= '1';
                            m_raddr <= counter2addr(counter_reg, rfskip_log);
                        end if;

                        rfts_next      <= ts;
                        stop           <= do_stop(counter_reg, '1', rfskip_log, rfdist);
                        refresh_strobe <= '1';
                        first_next     <= '1';
                        state_next     <= REFRESH;
                    end if;
                else
                    transparent := '1';
                end if;

            when REFRESH =>
                if READWRITE then
                    if ren_q = '1' and m_rack = '0' then
                        m_raddr <= counter2addr(counter_reg, rfskip_log);
                    end if;
                    if wen_q = '1' and m_wack = '0' then
                        m_waddr <= counter2addr(counter_reg - 1, rfskip_log);
                        m_wdata <= data_reg;
                    end if;

                    if m_rack = '1' then
                        ren       <= '0';
                        data_next <= m_rdata;
                    end if;
                    if m_wack = '1' then
                        wen <= '0';
                    end if;

                    if (ren_q = '0' or m_rack = '1') and (wen_q = '0' or m_wack = '1') then
                        if is_last(counter_reg, first_reg, rfskip_log, rfdist) = '1' then
                            transparent := '1';
                            data_next   <= (others => 'X');
                            state_next  <= NORMAL;

                            if counter_reg(ROWS_LOG) = '1' then
                                counter_next <= (others => '0');
                            end if;
                        else
                            wen     <= '1';
                            m_waddr <= counter2addr(counter_reg, rfskip_log);
                            if m_rack = '1' then
                                m_wdata <= m_rdata;
                            else
                                m_wdata <= data_reg;
                            end if;

                            if is_last(counter_reg + 1, '0', rfskip_log, rfdist) = '0' then
                                ren     <= '1';
                                m_raddr <= counter2addr(counter_reg + 1, rfskip_log);
                            end if;

                            counter_next <= counter_reg + 1;
                            first_next   <= '0';
                            stop         <= do_stop(counter_reg + 1, '0', rfskip_log, rfdist);
                        end if;
                    else
                        stop <= do_stop(counter_reg, first_reg, rfskip_log, rfdist);
                    end if;
                else
                    if is_last(counter_reg, first_reg, rfskip_log, rfdist) = '1' then
                        transparent := '1';
                        data_next   <= (others => 'X');
                        state_next  <= NORMAL;

                        if counter_reg(ROWS_LOG) = '1' then
                            counter_next <= (others => '0');
                        end if;
                    else
                        counter_next <= counter_reg + 1;
                        first_next   <= '0';
                        stop         <= do_stop(counter_reg + 1, '0', rfskip_log, rfdist);
                    end if;
                end if;

        end case;

        if transparent = '1' then
            m_raddr <= s_raddr;
            m_waddr <= s_waddr;
            m_wdata <= s_wdata;
            s_ready <= m_ready;

            if m_ready = '1' then
                if s_ren = '1' then
                    ren <= '1';
                end if;
                if s_wen = '1' then
                    wen <= '1';
                end if;
            end if;
        end if;

    end process;

    m_ren <= ren;
    m_wen <= wen;

end architecture rtl;
