library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_textio.all;
use std.textio.all;

entity memarray_ts is
    generic(
        WORDSIZE_LOG : natural;
        ROWS_LOG     : natural;
        TIMERSIZE    : natural;
        TDIFFSIZE    : natural;
        DEBUG        : boolean := false
    );
    port(
        clk          : in  std_logic;
        rst          : in  std_logic;

        -- Slave port
        raddr        : in  std_logic_vector(ROWS_LOG - 1 downto 0);
        ren          : in  std_logic;
        rdata        : out std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        rack         : out std_logic;
        waddr        : in  std_logic_vector(ROWS_LOG - 1 downto 0);
        wen          : in  std_logic;
        wdata        : in  std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        wack         : out std_logic;
        ready        : out std_logic;

        -- Modifier port
        mod_en       : out std_logic;
        mod_addr     : out std_logic_vector(ROWS_LOG - 1 downto 0);
        mod_tdiff    : out std_logic_vector(TDIFFSIZE - 1 downto 0);
        mod_data_in  : out std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        mod_data_out : in  std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        mod_ack      : in  std_logic;

        -- Control input
        ts           : in  std_logic_vector(TIMERSIZE - 1 downto 0)
    );
end entity memarray_ts;

architecture rtl of memarray_ts is
    constant WORDSIZE : natural := 2 ** WORDSIZE_LOG;
    type state_type is (IDLE, READ, MODIFY);

    signal state_reg, state_next                    : state_type;
    signal ram_ren, ram_wen                         : std_logic;
    signal ram_raddr, ram_waddr                     : std_logic_vector(ROWS_LOG - 1 downto 0);
    signal ram_rdata, ram_wdata, write_data_ts      : std_logic_vector(WORDSIZE + TIMERSIZE - 1 downto 0);
    signal wack_next                                : std_logic;
    signal raddr_reg, raddr_next                    : std_logic_vector(ROWS_LOG - 1 downto 0);
    signal read_data, read_data_reg, read_data_next : std_logic_vector(WORDSIZE - 1 downto 0);
    signal tsdiff                                   : std_logic_vector(TIMERSIZE - 1 downto 0);
    signal tdiff, tdiff_reg, tdiff_next             : std_logic_vector(TDIFFSIZE - 1 downto 0);
begin
    RAM : entity work.bram
        generic map(
            DWIDTH   => WORDSIZE + TIMERSIZE,
            AWIDTH   => ROWS_LOG,
            INITFILE => ""
        )
        port map(
            clk   => clk,
            rst   => rst,
            raddr => ram_raddr,
            ren   => ram_ren,
            rdata => ram_rdata,
            waddr => ram_waddr,
            wen   => ram_wen,
            wdata => ram_wdata
        );

    process(clk, rst) is
        variable debug_buf : line;
    begin
        if rst = '1' then
            state_reg     <= IDLE;
            raddr_reg     <= (others => 'X');
            read_data_reg <= (others => 'X');
            tdiff_reg     <= (others => 'X');
            wack          <= '0';
        elsif rising_edge(clk) then
            if DEBUG and state_next = READ then
                write(debug_buf, string'("[MARR] read 0x"));
                hwrite(debug_buf, raddr);
                writeline(output, debug_buf);
            end if;
            if DEBUG and wack_next = '1' then
                write(debug_buf, string'("[MARR] write 0x"));
                hwrite(debug_buf, waddr);
                writeline(output, debug_buf);
            end if;

            state_reg     <= state_next;
            raddr_reg     <= raddr_next;
            read_data_reg <= read_data_next;
            tdiff_reg     <= tdiff_next;
            wack          <= wack_next;
        end if;
    end process;

    process(state_reg, raddr_reg, read_data_reg, tdiff_reg, write_data_ts, read_data, tdiff, raddr, ren, waddr, wen, mod_data_out, mod_ack) is
        variable done : std_logic;
    begin
        done := '0';

        state_next     <= state_reg;
        raddr_next     <= raddr_reg;
        read_data_next <= read_data_reg;
        tdiff_next     <= tdiff_reg;
        wack_next      <= '0';

        rdata <= (others => 'X');
        rack  <= '0';
        ready <= '0';

        mod_en      <= '0';
        mod_addr    <= (others => 'X');
        mod_tdiff   <= (others => 'X');
        mod_data_in <= (others => 'X');

        ram_ren   <= '0';
        ram_wen   <= '0';
        ram_raddr <= (others => 'X');
        ram_waddr <= (others => 'X');
        ram_wdata <= (others => 'X');

        case state_reg is
            when IDLE =>
                -- see below

            when READ =>
                mod_en      <= '1';
                mod_addr    <= raddr_reg;
                mod_data_in <= read_data;
                mod_tdiff   <= tdiff;

                read_data_next <= read_data;
                tdiff_next     <= tdiff;
                state_next     <= MODIFY;

            when MODIFY =>
                if mod_ack = '0' then
                    mod_en      <= '1';
                    mod_addr    <= raddr_reg;
                    mod_data_in <= read_data_reg;
                    mod_tdiff   <= tdiff_reg;
                else
                    rdata <= mod_data_out;
                    done  := '1';

                    raddr_next     <= (others => 'X');
                    read_data_next <= (others => 'X');
                    tdiff_next     <= (others => 'X');
                    state_next     <= IDLE;
                end if;

        end case;

        rack <= done;

        if state_reg = IDLE or done = '1' then
            ready     <= '1';
            ram_raddr <= raddr;
            ram_ren   <= ren;
            ram_waddr <= waddr;
            ram_wen   <= wen;
            ram_wdata <= write_data_ts;
            wack_next <= wen;

            if ren = '1' then
                raddr_next <= raddr;
                state_next <= READ;
            end if;
        end if;

    end process;

    write_data_ts <= wdata & ts;
    read_data     <= ram_rdata(WORDSIZE + TIMERSIZE - 1 downto TIMERSIZE);
    tsdiff        <= std_logic_vector(unsigned(ts) - unsigned(ram_rdata(TIMERSIZE - 1 downto 0)));
    tdiff         <= tsdiff(TDIFFSIZE - 1 downto 0) when tsdiff(TIMERSIZE - 1 downto TDIFFSIZE) = (TIMERSIZE - 1 downto TDIFFSIZE => '0') else
                     (others => '1');

end architecture rtl;
