library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_textio.all;
use std.textio.all;

entity mem_slave is
    generic(
        AWIDTH       : natural;
        DWIDTH_LOG   : natural;

        WORDSIZE_LOG : natural;
        ROWS_LOG     : natural;
        DEBUG        : boolean := false
    );
    port(
        clk          : in  std_logic;
        rst          : in  std_logic;

        -- input port
        en           : in  std_logic;
        rnw          : in  std_logic;
        addr         : in  std_logic_vector(AWIDTH - 1 downto 0);
        wdata        : in  std_logic_vector(2 ** DWIDTH_LOG - 1 downto 0);
        be           : in  std_logic_vector(2 ** DWIDTH_LOG / 8 - 1 downto 0);
        rdata        : out std_logic_vector(2 ** DWIDTH_LOG - 1 downto 0);
        ack          : out std_logic;

        -- Master port
        m_raddr      : out std_logic_vector(ROWS_LOG - 1 downto 0);
        m_ren        : out std_logic;
        m_rdata      : in  std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        m_rack       : in  std_logic;
        m_waddr      : out std_logic_vector(ROWS_LOG - 1 downto 0);
        m_wen        : out std_logic;
        m_wdata      : out std_logic_vector(2 ** WORDSIZE_LOG - 1 downto 0);
        m_wack       : in  std_logic;

        -- Control output port
        read_strobe  : out std_logic;
        write_strobe : out std_logic
    );
end mem_slave;

architecture imp of mem_slave is
    constant DWIDTH   : natural := 2 ** DWIDTH_LOG;
    constant WORDSIZE : natural := 2 ** WORDSIZE_LOG;
    type state_type is (IDLE, READ, WRITE_R, WRITE_W);

    signal state_reg, state_next : state_type;
    signal row_reg, row_next     : std_logic_vector(ROWS_LOG - 1 downto 0);
    signal word_reg, word_next   : std_logic_vector(WORDSIZE_LOG - DWIDTH_LOG - 1 downto 0);
    signal data_reg, data_next   : std_logic_vector(WORDSIZE - 1 downto 0);
    signal wbus_reg, wbus_next   : std_logic_vector(WORDSIZE - 1 downto 0);
    signal wmask_reg, wmask_next : std_logic_vector(WORDSIZE - 1 downto 0);
begin
    process(clk, rst) is
        variable debug_buf : line;
    begin
        if rst = '1' then
            state_reg <= IDLE;
            row_reg   <= (others => 'X');
            word_reg  <= (others => 'X');
            data_reg  <= (others => 'X');
            wbus_reg  <= (others => 'X');
            wmask_reg <= (others => 'X');
        elsif rising_edge(clk) then
            if DEBUG and state_next = READ and (state_reg /= READ or m_rack = '1') then
                write(debug_buf, string'("[MEM ] read 0x"));
                hwrite(debug_buf, addr);
                writeline(output, debug_buf);
            end if;
            if DEBUG and ((state_next = WRITE_R and state_reg /= WRITE_R) or (state_next = WRITE_W and state_reg /= WRITE_R and (state_reg /= WRITE_W or m_wack = '1'))) then
                write(debug_buf, string'("[MEM ] write 0x"));
                hwrite(debug_buf, addr);
                writeline(output, debug_buf);
            end if;

            state_reg <= state_next;
            row_reg   <= row_next;
            word_reg  <= word_next;
            data_reg  <= data_next;
            wbus_reg  <= wbus_next;
            wmask_reg <= wmask_next;
        end if;
    end process;

    process(state_reg, row_reg, word_reg, data_reg, wbus_reg, wmask_reg, addr, be, en, rnw, wdata, m_rdata, m_rack, m_wack) is
        variable row  : std_logic_vector(ROWS_LOG - 1 downto 0);
        variable word : std_logic_vector(WORDSIZE_LOG - DWIDTH_LOG - 1 downto 0);
        variable done : std_logic;

        variable word_int     : integer;
        variable word_reg_int : integer;
    begin
        row  := addr(ROWS_LOG + WORDSIZE_LOG - 3 - 1 downto WORDSIZE_LOG - 3);
        word := addr(WORDSIZE_LOG - 3 - 1 downto DWIDTH_LOG - 3);
        done := '0';

        word_int     := 0;
        word_reg_int := 0;
        if word'length /= 0 and not is_X(word) then
            word_int := to_integer(unsigned(word));
        end if;
        if word_reg'length /= 0 and not is_X(word_reg) then
            word_reg_int := to_integer(unsigned(word_reg));
        end if;

        state_next <= state_reg;
        row_next   <= row_reg;
        word_next  <= word_reg;
        data_next  <= data_reg;
        wbus_next  <= wbus_reg;
        wmask_next <= wmask_reg;

        rdata <= (others => 'X');

        m_raddr <= (others => 'X');
        m_ren   <= '0';
        m_waddr <= (others => 'X');
        m_wen   <= '0';
        m_wdata <= (others => 'X');

        read_strobe  <= '0';
        write_strobe <= '0';

        case state_reg is
            when IDLE =>
                -- see below

            when READ =>
                if m_rack = '0' then
                    m_ren   <= '1';
                    m_raddr <= row_reg;
                else
                    done       := '1';
                    rdata      <= m_rdata((word_reg_int + 1) * DWIDTH - 1 downto word_reg_int * DWIDTH);
                    row_next   <= (others => 'X');
                    word_next  <= (others => 'X');
                    state_next <= IDLE;
                end if;

            when WRITE_R =>
                if m_rack = '0' then
                    m_ren   <= '1';
                    m_raddr <= row_reg;
                else
                    m_wen      <= '1';
                    m_waddr    <= row_reg;
                    m_wdata    <= (m_rdata and not wmask_reg) or (wbus_reg and wmask_reg);
                    data_next  <= m_rdata;
                    state_next <= WRITE_W;
                end if;

            when WRITE_W =>
                if m_wack = '0' then
                    m_wen   <= '1';
                    m_waddr <= row_reg;
                    m_wdata <= (data_reg and not wmask_reg) or (wbus_reg and wmask_reg);
                else
                    done       := '1';
                    row_next   <= (others => 'X');
                    word_next  <= (others => 'X');
                    wbus_next  <= (others => 'X');
                    wmask_next <= (others => 'X');
                    data_next  <= (others => 'X');
                    state_next <= IDLE;
                end if;

        end case;

        ack <= done;

        if (state_reg = IDLE or done = '1') and en = '1' then
            row_next  <= row;
            word_next <= word;

            if rnw = '1' then
                m_ren       <= '1';
                m_raddr     <= row;
                state_next  <= READ;
                read_strobe <= '1';
            else
                wbus_next                                                       <= (others => '0');
                wbus_next((word_int + 1) * DWIDTH - 1 downto word_int * DWIDTH) <= wdata;

                wmask_next <= (others => '0');
                for i in be'range loop
                    wmask_next(word_int * DWIDTH + 8 * (i + 1) - 1 downto word_int * DWIDTH + 8 * i) <= (others => be(i));
                end loop;

                if DWIDTH = WORDSIZE and be = (be'range => '1') then
                    m_wen      <= '1';
                    m_waddr    <= row;
                    m_wdata    <= wdata;
                    data_next  <= (others => '0');
                    state_next <= WRITE_W;
                else
                    m_ren      <= '1';
                    m_raddr    <= row;
                    state_next <= WRITE_R;
                end if;
                write_strobe <= '1';
            end if;
        end if;
    end process;

end architecture imp;
