library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_textio.all;
use std.textio.all;

entity ipic_slave is
    generic(
        C_IPIC_AWIDTH  : integer;
        C_IPIC_DWIDTH  : integer;
        C_IPIC_NUM_REG : integer;
        C_IPIC_NUM_MEM : integer;
        DEBUG          : boolean := false
    );
    port(
        clk          : in  std_logic;
        rst          : in  std_logic;

        -- IPIC port
        Bus2IP_Addr  : in  std_logic_vector(C_IPIC_AWIDTH - 1 downto 0);
        Bus2IP_CS    : in  std_logic_vector(0 to C_IPIC_NUM_MEM - 1);
        Bus2IP_RNW   : in  std_logic;
        Bus2IP_Data  : in  std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);
        Bus2IP_BE    : in  std_logic_vector(C_IPIC_DWIDTH / 8 - 1 downto 0);
        Bus2IP_RdCE  : in  std_logic_vector(0 to C_IPIC_NUM_REG - 1);
        Bus2IP_WrCE  : in  std_logic_vector(0 to C_IPIC_NUM_REG - 1);
        IP2Bus_Data  : out std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);
        IP2Bus_RdAck : out std_logic;
        IP2Bus_WrAck : out std_logic;
        IP2Bus_Error : out std_logic;

        -- Register port
        regs_rdata   : in  std_logic_vector(C_IPIC_NUM_REG * C_IPIC_DWIDTH - 1 downto 0);
        regs_wen     : out std_logic_vector(C_IPIC_NUM_REG - 1 downto 0);
        regs_wdata   : out std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);

        -- Memory port
        mem_en       : out std_logic_vector(C_IPIC_NUM_MEM - 1 downto 0);
        mem_rnw      : out std_logic;
        mem_addr     : out std_logic_vector(C_IPIC_AWIDTH - 1 downto 0);
        mem_wdata    : out std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);
        mem_be       : out std_logic_vector(C_IPIC_DWIDTH / 8 - 1 downto 0);
        mem_rdata    : in  std_logic_vector(C_IPIC_NUM_MEM * C_IPIC_DWIDTH - 1 downto 0);
        mem_ack      : in  std_logic_vector(C_IPIC_NUM_MEM - 1 downto 0)
    );
end entity ipic_slave;

architecture imp of ipic_slave is
    signal regs_read_ack, regs_write_ack : std_logic;
    signal regs_data                     : std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);

    signal mem_read_ack, mem_write_ack : std_logic;
    signal mem_data                    : std_logic_vector(C_IPIC_DWIDTH - 1 downto 0);
begin
    regs_write_ack <= '1' when Bus2IP_WrCE /= (Bus2IP_WrCE'range => '0') else '0';
    regs_read_ack  <= '1' when Bus2IP_RdCE /= (Bus2IP_RdCE'range => '0') else '0';

    process(clk, rst) is
        variable debug_buf : line;
    begin
        if rst = '1' then
        elsif rising_edge(clk) then
            for i in 0 to C_IPIC_NUM_REG - 1 loop
                if DEBUG and Bus2IP_RdCE(i) = '1' then
                    write(debug_buf, string'("[PLB ] read reg "));
                    write(debug_buf, i);
                    writeline(output, debug_buf);
                end if;
                if DEBUG and Bus2IP_WrCE(i) = '1' then
                    write(debug_buf, string'("[PLB ] write reg "));
                    write(debug_buf, i);
                    write(debug_buf, string'(" data=0x"));
                    hwrite(debug_buf, Bus2IP_Data);
                    writeline(output, debug_buf);
                end if;
            end loop;
        end if;
    end process;

    process(Bus2IP_RdCE, Bus2IP_WrCE, Bus2IP_BE, Bus2IP_Data, regs_rdata) is
    begin
        regs_data  <= (others => 'X');
        regs_wdata <= (others => 'X');
        for i in 0 to C_IPIC_NUM_REG - 1 loop
            if Bus2IP_RdCE(i) = '1' then
                regs_data <= regs_rdata((i + 1) * C_IPIC_DWIDTH - 1 downto i * C_IPIC_DWIDTH);
            end if;
            if Bus2IP_WrCE(i) = '1' then
                regs_wdata <= regs_rdata((i + 1) * C_IPIC_DWIDTH - 1 downto i * C_IPIC_DWIDTH);
                for j in Bus2IP_BE'range loop
                    if Bus2IP_BE(j) = '1' then
                        regs_wdata(8 * (j + 1) - 1 downto 8 * j) <= Bus2IP_Data(8 * (j + 1) - 1 downto 8 * j);
                    end if;
                end loop;
            end if;
        end loop;
    end process;

    REG_GEN : for i in 0 to C_IPIC_NUM_REG - 1 generate
        regs_wen(i) <= Bus2IP_WrCE(i);
    end generate;


    mem_read_ack  <= '1' when Bus2IP_RNW = '1' and mem_ack /= (mem_ack'range => '0') else '0';
    mem_write_ack <= '1' when Bus2IP_RNW = '0' and mem_ack /= (mem_ack'range => '0') else '0';

    process(clk, rst) is
        variable debug_buf : line;
    begin
        if rst = '1' then
        elsif rising_edge(clk) then
            for i in 0 to C_IPIC_NUM_MEM - 1 loop
                if DEBUG and Bus2IP_CS(i) = '1' and Bus2IP_RNW = '1' and mem_ack(i) = '1' then
                    write(debug_buf, string'("[PLB ] read mem "));
                    write(debug_buf, i);
                    write(debug_buf, string'(" addr=0x"));
                    hwrite(debug_buf, Bus2IP_Addr);
                    writeline(output, debug_buf);
                end if;
                if DEBUG and Bus2IP_CS(i) = '1' and Bus2IP_RNW = '0' and mem_ack(i) = '1' then
                    write(debug_buf, string'("[PLB ] write mem "));
                    write(debug_buf, i);
                    write(debug_buf, string'(" addr=0x"));
                    hwrite(debug_buf, Bus2IP_Addr);
                    write(debug_buf, string'(" data=0x"));
                    hwrite(debug_buf, Bus2IP_Data);
                    writeline(output, debug_buf);
                end if;
            end loop;
        end if;
    end process;

    mem_addr  <= Bus2IP_Addr;
    mem_wdata <= Bus2IP_Data;

    process(Bus2IP_CS, Bus2IP_RNW, mem_rdata) is
    begin
        mem_data <= (others => 'X');
        for i in 0 to C_IPIC_NUM_MEM - 1 loop
            if Bus2IP_CS(i) = '1' and Bus2IP_RNW = '1' then
                mem_data <= mem_rdata((i + 1) * C_IPIC_DWIDTH - 1 downto i * C_IPIC_DWIDTH);
            end if;
        end loop;
    end process;

    mem_rnw <= Bus2IP_RNW;
    mem_be  <= Bus2IP_BE;

    MEM_GEN : for i in 0 to C_IPIC_NUM_MEM - 1 generate
        mem_en(i) <= Bus2IP_CS(i) and not mem_ack(i);
    end generate;

    IP2Bus_Data <= regs_data when regs_read_ack = '1' else
                   mem_data when mem_read_ack = '1' else
                   (others => 'X');

    IP2Bus_WrAck <= regs_write_ack or mem_write_ack;
    IP2Bus_RdAck <= regs_read_ack or mem_read_ack;
    IP2Bus_Error <= '0';

end imp;
