"""File for initialization of Maxwell equations problem"""

import numpy as np
from scipy.optimize import fsolve

def CavityExact(x, epl, epr, mul, mur, t):
    """Purpose: Set up exact solution to EM cavity problem"""

    xL = len(x)
    n1 = np.sqrt(epl*mul)
    n2 = np.sqrt(epr*mur)

    qf = np.zeros((2,xL))

    ep = np.zeros(xL)
    mu = np.zeros(xL)

    # Compute omega to match coefficients - set initial guess to obtain 
    # different solutions

    f = lambda x: n1*np.tan(n2*x) + n2*np.tan(n1*x)
    omega = fsolve( f, 6.0)

    # Set up exact solution
    A1 = np.complex(n2*np.cos(omega*n2)/(n1*np.cos(n1*omega)))
    A2 = np.exp(-1.0j*omega*(n1+n2))
    B1 = A1*np.exp(-1.0j*2*n1*omega)
    B2 = A2*np.exp(1.0j*2*n2*omega)

    for i in range(xL):
        if x[i] <= 0.0:
            A = A1
            B = B1
            n = n1

            ep[i] = epl
            mu[i] = mul
        else:
            A = A2
            B = B2
            n = n2

            ep[i] = epr
            mu[i] = mur

        Eh = (A*np.exp(1.0j*n*omega*x[i]) - \
                B*np.exp(-1.0j*n*omega*x[i]))*np.exp(-1.0j*omega*t)

        Hh = n*(A*np.exp(1.0j*n*omega*x[i]) + \
                B*np.exp(-1.0j*n*omega*x[i]))*np.exp(-1.0j*omega*t)

        qf[0,i] = np.real(Eh)
        qf[1,i] = np.real(Hh)

    return qf,ep,mu


def CavityExactLong(x, epl, epr, mul, mur, t):
    """Purpose: Set up exact solution to EM cavity problem [-2,2]
    NOTE: n1=1 is assumed"""

    xL = len(x)
    n1 = 1.0
    n2 = np.sqrt(epr*mur)

    qf = np.zeros((2,xL))

    ep = np.zeros(xL)
    mu = np.zeros(xL)

    # Compute omega to match coefficients - set initial guess to obtain 
    # different solutions

    f = lambda x: n1*np.tan(n2*x) + n2*np.tan(n1*x)
    omega = fsolve( f, 6.0)

    # Set up exact solution
    A1 = np.complex(n2*np.cos(omega*n2)/(n1*np.cos(n1*omega)))
    A2 = np.exp(-1.0j*omega*2)
    A3 = np.exp(-4.0j*omega)*A1

    B1 = A3
    B2 = A2
    B3 = A1

    for i in range(xL):
        if x[i] <= -1.0:
            A = A1
            B = B1
            n = n1

            ep[i] = epl
            mu[i] = mul

        elif x[i] >=1.0:
            A = A3
            B = B3
            n = n1

            ep[i] = epl
            mu[i] = mul

        else:
            A = A2
            B = B2
            n = n2

            ep[i] = epr
            mu[i] = mur

        Eh = (A*np.exp(1.0j*n*omega*x[i]) - \
                B*np.exp(-1.0j*n*omega*x[i]))*np.exp(1.0j*omega*t)

        Hh = n*(A*np.exp(1.0j*n*omega*x[i]) + \
                B*np.exp(-1.0j*n*omega*x[i]))*np.exp(1.0j*omega*t)

        qf[0,i] = np.real(Eh)
        qf[1,i] = np.real(Hh)

    return qf,ep,mu
