"""Driver script for solving the 1D Maxwell equations using a DG scheme"""

import numpy as np
import matplotlib.pyplot as plt
from MaxwellDG import *
from MaxwellIC import CavityExact
from Legendre import LegendreGL

# Order of the method (m), number of elements (N)
m = 4
N = 10

# Set problem parameters
CFL = 0.25
FinalTime = np.sqrt(2.0)
epl = 1.0
mul = 1.0
epr = 2.0
mur = 1.0

# Define domain, materials and initial conditions
xmin = -1.0
xmax = 1.0
L = xmax-xmin

# Generate mesh
VX = xmin + (xmax-xmin)/N*np.arange(N+1)
r,w = LegendreGL(m) 
x = np.outer( np.ones(m+1), VX[:N] ) + 0.5*np.outer(r+1.0, VX[1:N+1] - VX[:N] )
h = (xmax-xmin)/N

# Define domain,materials and initial conditions
q = np.zeros((m+1,2,N))
mu = np.zeros((m+1,N))
ep = np.zeros((m+1,N))

for k in range(N):
	a, ep[:,k], mu[:,k] = CavityExact(x[:,k],epl,epr,mul,mur,0.0)
	q[:,:,k] = a.T

# Set up material parameters
eps1 = np.concatenate((epl*np.ones(int(N/2)), epr*np.ones(int(N/2))))
mu1 = np.concatenate((mul*np.ones(int(N/2)), mur*np.ones(int(N/2))))

ep = np.outer(np.ones(m+1), eps1)
mu = np.outer(np.ones(m+1), mu1)

# Solve Problem
q = MaxwellDG1D(x,q,ep,mu,h,m,N,CFL,FinalTime)

# Plot
plt.plot(x[0,:],q[0,0,:])
plt.title("Solution E at time {}".format(FinalTime))
plt.show()