"""
This file contains the specific functions to solve Maxwell equations
in 1D using a discontinuous galerkin scheme.
"""

import numpy as np
from Maxwell import *
from DiscontinuousGalerkin import *
from helpers import extendDG

def MaxwellDGrhs1D(x,q,ep,mu,h,k,m,N,Ma,S,VtoE,maxvel):

    """Purpose  : Evaluate the RHS of Maxwell equations using a DG method"""
    Imat = np.eye(m+1)
    qe = np.zeros((2,2,N+2))

    E = q[:,0,:]
    H = q[:,1,:]

    # Extend data and assign boundary conditions
    qe[:,0,:] = extendDG(E[VtoE[0],VtoE[1]], "D", 0.0, "D", 0.0)
    qe[:,1,:] = extendDG(H[VtoE[0],VtoE[1]], "N", 0, "N", 0) 

    # Compute numerical fluxes at interfaces
    fluxr = MaxwellLF( qe[1,:,1:N+1], qe[0,:,2:N+2], ep[0,:],mu[0,:], maxvel)
    fluxl = MaxwellLF( qe[1,:,:N], qe[0,:,1:N+1], ep[0,:], mu[0,:], maxvel)

    # Compute right hand side of Maxwell's equation
    rE = np.dot(S.T, H/ep) - (np.outer(Imat[:,m], fluxr[0,:]) - \
                np.outer(Imat[:,0], fluxl[0,:]))
    rH = np.dot(S.T, E/mu) - (np.outer(Imat[:,m], fluxr[1,:]) - \
                np.outer(Imat[:,0], fluxl[1,:]))

    rhsq = np.zeros(q.shape)
    rhsq[:,0,:] = np.linalg.solve(h/2*Ma, rE )
    rhsq[:,1,:] = np.linalg.solve(h/2*Ma, rH )
    return rhsq

def MaxwellDG1D(x,q,ep,mu,h,m,N,CFL,FinalTime):
    """Purpose  : Integrate 1D Maxwell equation until FinalTime using a DG
                  scheme and 3rd order SSP-RK method
    """   

    # Initialize operators at Legendre Gauss Lobatto grid
    r,w = LegendreGL(m)
    V = VandermondeDG(m,r)
    D = DmatrixDG(m, r, V)

    Ma = np.linalg.inv(np.dot(V,np.transpose(V)))
    S = np.dot(Ma,D)
    iV = np.linalg.inv(V)

    # Compute operator for WENO smoothness evaluator
    Q,Xm,Xp = WENODGWeights(m,iV)

    # Initialize extraction vector
    VtoE = [np.array(([0]*N, [m]*N)) , np.arange(N)]

    # Compute smallest spatial scale timestep
    rLGLmin = np.abs(r[0]-r[1])

    t = 0.0
    tstep = 0

    # Set timestep
    cvel = 1.0/np.sqrt(ep*mu)
    maxvel = cvel.max()
    k = CFL*rLGLmin*h/2/maxvel

    #integrate scheme
    while (t<FinalTime):
        # Decide on timestep
        k = min(k,FinalTime-t)
        # Update solution
        rhsq  = MaxwellDGrhs1D(x,q,ep,mu,h,k,m,N,Ma,S,VtoE,maxvel)
        q1 = q + k*rhsq

        rhsq  = MaxwellDGrhs1D(x,q1,ep,mu,h,k,m,N,Ma,S,VtoE,maxvel) 
        q2 = (3*q + q1 + k*rhsq)/4

        rhsq  = MaxwellDGrhs1D(x,q2,ep,mu,h,k,m,N,Ma,S,VtoE,maxvel)
        q = (q + 2*q2 + 2*k*rhsq)/3
        
        t += k
        tstep += 1

    return q
