"""
This file contains the specific functions to solve linear wave equation
in 1D or 2D using a slope limited scheme.
"""

import numpy as np
from Linwave import *
from helpers import extend,SlopeLimit

## 1D ##
###################################################################################
def LinwaveSLrhs1D(x, u, h, k, maxvel):

    """Purpose: Evaluate right hand side for Linwave equation using a
                slope limited scheme"""
    N = len(x)

    # Chose slope limiter - 0:LF; 1:minmod; 2:MUSCL; 3:Superbee; 
    # 4:van Albada; 5:van Leer, 6: TVB 
    typ = 2
    c = 0.0
    M = 10.0

    # Boundary conditions 
    xe, ue = extend(x, u, 2, "P", 0, "P", 0)

    # Compute element slope and limit
    dup = ue[2:N+4]-ue[1:N+3]
    dum = ue[1:N+3]-ue[:N+2]
    duL = SlopeLimit(dup,dum,typ,c,M,h)

    # Compute cell interface values - for f'(u) = 1
    uh = ue[1:N+3] - k/(2.0*h)*duL
    uL = uh - 0.5*duL
    uR = uh + 0.5*duL


    # Compute RHS
    du = - (LinwaveLF(uR[1:N+1], uL[2:N+2], maxvel) - \
            LinwaveLF(uR[:N], uL[1:N+1],maxvel))/h
    return du

def LinwaveSL1D(x, u, h, CFL, FinalTime):
    """Purpose  : Integrate 1D wave equation until FinalTime using a slope limited scheme.
    """   
    t = 0.0
    timestep = 0

    # Set timestep
    k = CFL*h
    maxvel = 1.0

    while t < FinalTime:
        k = min(FinalTime-t, k)
        # Update solution
        u += k*LinwaveSLrhs1D(x,u,h,k,maxvel)
        t +=k
        timestep += 1
        
    return u