"""
This file contains the specific functions to solve the linear wave equation
in 1D using a central scheme.
"""

import numpy as np
from Linwave import *
from helpers import extend, extendstag, SlopeLimit

###################################################################################
def LinwaveCrhs1D(x, u, h, k):
    """Purpose: Evaluate right hand side for Linwave equation using second order
       central scheme - NOTE two steps of k/2 is taken"""

    N = len(u)
    duL = np.zeros(N+2)
    uh = np.zeros(N+2)
    
    # choose limiter type:
    # 0: LF; 1: minmod; 2: MUSCL; 3: Superbee;
    # 4:van Albada; 5:van Leer; 6: TVB
    typ = 2
    c = 0.0
    M = 10.0

    # First step from non-staggered to staggered grid
    xe, ue = extend(x, u, 2, "P", 0, "P", 0) #periodic BC


    # Compute element slope and limit
    dup = ue[2:N+4] - ue[1:N+3]
    dum = ue[1:N+3] - ue[:N+2]
    duL = SlopeLimit(dum, dup, typ, c, M, h)

    # Compute limited intermediate solution
    uh = ue[1:N+3] - k/(4*h)*duL

    # Advance solution k/2
    us = 0.5*(u[:N-1] + u[1:N]) + 1.0/8*(duL[1:N] - duL[2:N+1]) - \
    	 k/2/h*((uh[2:N+1]) - (uh[1:N]))

    # Second step from staggered to non-staggered grid
    Ns = N-1
    duL = np.zeros(Ns+2)
    
    xe, ue = extendstag(x[:Ns], us, 2, "P", 0, "P", 0) # periodic BC

    # Compute element slope and limit 
    dup = ue[2:Ns+4] - ue[1:Ns+3]
    dum = ue[1:Ns+3] - ue[:Ns+2]
    duL = SlopeLimit(dum,dup, typ, c, M, h)
    
    # Compute limited intermediate solution
    uh = ue[1:Ns+3] - k/(4*h)*duL

    # Advance solution k/2
    uns = 0.5*(ue[1:Ns+2] + ue[2:Ns+3]) + 1.0/8*(duL[:Ns+1] - duL[1:Ns+2]) - \
    	  k/2/h*((uh[1:Ns+2]) - (uh[:Ns+1]))
    
    # Restore residual
    du = (uns-u)/k

    return du


def LinwaveC1D(x, u, h, CFL, FinalTime):
    """Purpose: Integrate 1D Linwave equation until FinalTime using a second order central scheme"""
    
    t = 0.0
    tstep = 0

    # Set timestep
    k = CFL*h
    maxvel = 1.0

    while t < FinalTime:
        k = min(FinalTime-t,k)
        # Update solution
        u += k*LinwaveCrhs1D(x, u, h, k)
        t += k
        tstep += 1

    return u



