"""
This file contains the specific functions to solve KPP equations
in 2D using a monotone scheme.
"""

import numpy as np
from KPP import *
from helpers import extend

## 2D ##
###################################################################################
def KPPMrhs2D(x, y, u, hx, hy, k, maxvel):
    """Purpose: Evaluate right hand side for 2D KPP equation using a monotone scheme
    """   

    Ny,Nx = u.shape

    du = np.zeros((Ny,Nx))


    for i in range(Ny):
        xe, ue = extend(x[i,:], u[i,:], 1, "D", np.pi/4, "D", np.pi/4)
        du[i,:] = -1.0/hx* (KPPxLF(ue[1:-1], ue[2:], maxvel) - \
                              KPPxLF(ue[0:-2], ue[1:-1], maxvel) )

    for j in range(Nx):
        ye, ue = extend(y[:,j], u[:,j], 1, "D", np.pi/4, "D", np.pi/4)

        du[:,j] -= 1.0/hy* (KPPyLF(ue[1:-1], ue[2:], maxvel) - \
                              KPPyLF(ue[0:-2], ue[1:-1], maxvel) )
        
    return du
    

def KPPM2D(x, y, u, hx, hy, CFL, FinalTime):
    """Purpose: Integrate 2D KPP equation until FinalTime using a monotone scheme. 
    """   
    t = 0.0
    tstep = 0
    delta = min(hx,hy)
    maxvel = 1.0
    k = CFL*delta/maxvel/2
    
    # Integrate scheme
    while t < FinalTime:
        # Decide on timestep
        k = min(FinalTime-t, k)
        u += k*KPPMrhs2D(x, y, u, hx, hy, k, maxvel)
        t += k
        tstep += 1
        
    return u