"""
This file contains the specific functions to solve Euler equations
in 1D or 2D using a slope limited scheme.
"""

import numpy as np
from Euler import *
from helpers import extend,SlopeLimit

## 1D ##
###################################################################################
def EulerSLrhs1D(x,q,gamma,h,k,maxvel):

    """Purpose: Evaluate right hand side for Euler equation using a
                slope limited scheme"""
    N = len(x)
    qe = np.zeros((3,N+4))
    duL = np.zeros((3,N+2))

    # Chose slope limiter - 0:LF; 1:minmod; 2:MUSCL; 3:Superbee; 
    # 4:van Albada; 5:van Leer, 6: TVB 
    typ = 2
    c = h**3
    M = 150.0

    # Extend data and assign boundary conditions 
    xe, qe[0,:] = extend(x, q[0,:], 2, "D", 1.0, "D", 0.125)
    xe, qe[1,:] = extend(x, q[1,:], 2, "D", 0.0, "N", 0.0)
    xe, qe[2,:] = extend(x, q[2,:], 2, "D", 2.5, "N", 0.0)

    # xe, qe[0,:] = extend(x, q[0,:], 2, "D", 3.857143, "N", 0.0)
    # xe, qe[1,:] = extend(x, q[1,:], 2, "D", 10.141852, "D", 0.0)
    # xe, qe[2,:] = extend(x, q[2,:], 2, "D", 39.166661, "N", 0.0)


    # Compute left and right differences, evaluate slopes and interface values
    dup = qe[:,2:N+4]-qe[:,1:N+3]
    dum = qe[:,1:N+3]-qe[:,:N+2]
    duL[0,:] = SlopeLimit(dup[0,:],dum[0,:],typ,c,M,h)
    duL[1,:] = SlopeLimit(dup[1,:],dum[1,:],typ,c,M,h)
    duL[2,:] = SlopeLimit(dup[2,:],dum[2,:],typ,c,M,h)
    qL = qe[:,1:N+3]-0.5*duL
    qR = qe[:,1:N+3]+0.5*duL


    # Evaluate right hand side using numerical flux
    dq = - (EulerLF(qR[:,1:N+1], qL[:,2:N+2], gamma, maxvel) - \
            EulerLF(qR[:,:N], qL[:,1:N+1], gamma, maxvel))/h
    return dq

def EulerSLcharrhs1D(x,q,gamma,h,k,maxvel):

    """Purpose: Evaluate right hand side for Euler equation using a
                slope limited scheme with limiting on characteristic 
                variables"""
    N = len(x)
    qe = np.zeros((3,N+4))
    Lamq = np.zeros((3,N+4))
    qL = np.zeros((3,N+2))
    qR = np.zeros((3,N+2))
    dRlocL = np.zeros(3)
    RlocL = np.zeros((3,N+2))
    RlocR = np.zeros((3,N+2))

    # Chose slope limiter - 0:LF; 1:minmod; 2:MUSCL; 3:Superbee; 
    # 4:van Albada; 5:van Leer, 6: TVB 
    typ = 2
    c = 0.0
    M = 0.0

    # Extend data and assign boundary conditions 
    xe, qe[0,:] = extend(x, q[0,:], 2, "D", 1.0, "D", 0.125)
    xe, qe[1,:] = extend(x, q[1,:], 2, "D", 0.0, "N", 0.0)
    xe, qe[2,:] = extend(x, q[2,:], 2, "D", 2.5, "N", 0.0)

    # xe, qe[0,:] = extend(x, q[0,:], 2, "D", 3.857143, "N", 0.0)
    # xe, qe[1,:] = extend(x, q[1,:], 2, "D", 10.141852, "D", 0.0)
    # xe, qe[2,:] = extend(x, q[2,:], 2, "D", 39.166661, "N", 0.0)

    # Extract characteristic variables
    Rchar = np.zeros((3,N+4))
    for i in range(N+4):
        S,iS,Lam = EulerChar(qe[:,i],gamma)
        Rchar[:,i] = np.dot(iS, qe[:,i])
        Lamq[:,i] = np.diag(Lam)

    # Compute limited slopes and values at cell interfaces
    for i in range(N+2):
        Rloc = Rchar[:,i:i+3]
        Laml = Lamq[:,i+1]

        # Determine upwind directions
        for j in range(3):
            rRloc = (Rloc[j,1]-Rloc[j,0])/(Rloc[j,2]-Rloc[j,1])
            if Laml[j] >= 0.0: # Upwind
                dRlocL[j] = (Rloc[j,1]-Rloc[j,0])*0.0 # ? SlopeLimit(1.0/rRloc,typ,c,M,h)
            else: # Downwind
                dRlocL[j] = (Rloc[j,2]-Rloc[j,1])*0.0 # ? SlopeLimit(rRloc,typ,c,M,h)

        Rloch = Rloc[:,1] - k/(2.0*h)*(Laml*dRlocL)
        RlocL = Rloch - 0.5*dRlocL
        RlocR = Rloch + 0.5*dRlocL

        S,iS,Lam = EulerChar(qe[:,i+1],gamma)
        qL[:,i] = np.dot(S, RlocL)
        qR[:,i] = np.dot(S, RlocR)


    # Evaluate right hand side using numerical flux
    dq = - (EulerLF(qR[:,1:N+1], qL[:,2:N+2], gamma, maxvel) - \
            EulerLF(qR[:,:N], qL[:,1:N+1], gamma, maxvel))/h
    return dq

def EulerSL1D(x,q,h,CFL,gamma,FinalTime):
    """Purpose  : Integrate 1D Euler equation until FinalTime using a
                 slope limited scheme and a SSP-RK3.
    """   
    t = 0.0
    timestep = 0

    while t < FinalTime:
        # Set timestep
        p = (gamma-1.0)*(q[2,:]-0.5*q[1,:]**2/q[0,:])
        c = np.sqrt(gamma*p/q[0,:])
        maxvel = (c+np.abs(q[1,:]/q[0,:])).max()
        k = min(FinalTime-t, CFL*h/maxvel)

        # Update solution
        rhsq = EulerSLcharrhs1D(x,q,gamma,h,k,maxvel)
        q1 = q + k*rhsq

        rhsq  = EulerSLcharrhs1D(x,q1,gamma,h,k,maxvel)
        q2 = (3*q + q1 + k*rhsq)/4

        rhsq  = EulerSLcharrhs1D(x,q2,gamma,h,k,maxvel)
        q = (q + 2*q2 + 2*k*rhsq)/3

        t +=k
        timestep += 1
        
    return q

## 2D ##
###################################################################################
def EulerSLrhs2D(x,y,q,gamma,hx,hy,k):
    """Purpose: Evaluate right hand side for 2D Euler equation 
                using a slope limited scheme
    """   

    n, Ny,Nx = q.shape

    dq = np.zeros((4,Ny,Nx))
    qex = np.zeros((4,Nx+4))
    qey = np.zeros((4,Ny+4))
    dqLx = np.zeros((4,Nx+2))
    dqLy = np.zeros((4,Ny+2))

    # Chose slope limiter - 0:LF; 1:minmod; 2:MUSCL; 3:Superbee; 
    # 4:van Albada; 5:van Leer, 6: TVB 
    typ = 2
    c = 0.0
    M = 10.0

    # Apply slope limited scheme in the x direction
    for i in range(Ny):
        # Extend data and assign boundary conditions
        xe,qex[0,:] = extend(x[i,:], q[0,i,:], 2, "N", 0, "N", 0)
        xe,qex[1,:] = extend(x[i,:], q[1,i,:], 2, "N", 0, "N", 0)
        xe,qex[2,:] = extend(x[i,:], q[2,i,:], 2, "N", 0, "N", 0)
        xe,qex[3,:] = extend(x[i,:], q[3,i,:], 2, "N", 0, "N", 0)

        # Compute element slope and limit
        dup = qex[:,2:Nx+4]-qex[:,1:Nx+3]
        dum = qex[:,1:Nx+3]-qex[:,:Nx+2]
        dqLx[0,:] = SlopeLimit(dup[0,:],dum[0,:],typ,c,M,hx)
        dqLx[1,:] = SlopeLimit(dup[1,:],dum[1,:],typ,c,M,hx)
        dqLx[2,:] = SlopeLimit(dup[2,:],dum[2,:],typ,c,M,hx)
        dqLx[3,:] = SlopeLimit(dup[3,:],dum[3,:],typ,c,M,hx)

        ql = qex[:,1:Nx+3] - 0.5*dqLx
        qr = qex[:,1:Nx+3] + 0.5*dqLx

        # Compute fluxes
        dq1 = EulerLF2Dx( qr[:,1:Nx+1], ql[:,2:Nx+2], gamma )
        dq2 = EulerLF2Dx( qr[:,:Nx], ql[:,1:Nx+1], gamma )

        # Compute RHS
        dq[:,i,:] = - (dq1-dq2)/hx

    for j in range(Nx):
        # Extend data and assign boundary conditions
        xe,qey[0,:] = extend(y[:,j], q[0,:,j], 2, "N", 0, "N", 0)
        xe,qey[1,:] = extend(y[:,j], q[1,:,j], 2, "N", 0, "N", 0)
        xe,qey[2,:] = extend(y[:,j], q[2,:,j], 2, "N", 0, "N", 0)
        xe,qey[3,:] = extend(y[:,j], q[3,:,j], 2, "N", 0, "N", 0)

        # Compute element slope and limit
        dup = qey[:,2:Ny+4]-qey[:,1:Ny+3]
        dum = qey[:,1:Ny+3]-qey[:,:Ny+2]
        dqLy[0,:] = SlopeLimit(dup[0,:],dum[0,:],typ,c,M,hy)
        dqLy[1,:] = SlopeLimit(dup[1,:],dum[1,:],typ,c,M,hy)
        dqLy[2,:] = SlopeLimit(dup[2,:],dum[2,:],typ,c,M,hy)
        dqLy[3,:] = SlopeLimit(dup[3,:],dum[3,:],typ,c,M,hy)

        ql = qey[:,1:Ny+3] - 0.5*dqLy
        qr = qey[:,1:Ny+3] + 0.5*dqLy

        # Compute fluxes
        dq1 = EulerLF2Dy( qr[:,1:Ny+1], ql[:,2:Ny+2], gamma )
        dq2 = EulerLF2Dy( qr[:,:Ny], ql[:,1:Ny+1], gamma )

        # Compute RHS
        dq[:,:,j] -= (dq1-dq2)/hy
        
    return dq
    

def EulerSL2D(x,y,q,hx,hy,gamma,CFL,FinalTime):
    """Purpose  : Integrate 2D Euler equation until FinalTime using a slope limited scheme.
    """   
    t = 0.0
    tstep = 0
    delta = min(hx,hy)
    
    while t < FinalTime:
        p = (gamma-1.0)*(q[3,:,:]-0.5*(q[1,:,:]**2+q[2,:,:]**2)/q[0,:,:])
        c = np.sqrt(gamma*p/q[0,:,:])
        maxvelu = (c+np.abs(q[1,:,:]/q[0,:,:])).max()
        maxvelv = (c+np.abs(q[2,:,:]/q[0,:,:])).max()
        k = min(FinalTime-t, CFL*delta/np.sqrt(maxvelu**2+maxvelv**2)/np.sqrt(2.0))

        # Update solution using 3rd SSP-RK
        dq = EulerSLrhs2D(x,y,q,gamma,hx,hy,k)
        q1 = q + k*dq
        dq = EulerSLrhs2D(x,y,q1,gamma,hx,hy,k)
        q2 = (3*q+q1+k*dq)/4
        dq = EulerSLrhs2D(x,y,q2,gamma,hx,hy,k)
        q  = (q+2*q2+2*k*dq)/3

        t += k
        tstep += 1
        
    return q