"""
This file contains the specific functions to solve Euler equations
in 1D or 2D using a monotone scheme.
"""

import numpy as np
from Euler import *
from helpers import extend

## 1D ##
###################################################################################
def EulerM1Drhs(x,q,gamma,h,k,maxvel):
    """Purpose: Evaluate right hand side for the Euler equations
                using a monotone method
    """ 

    N = len(x)

    qe = np.zeros((3 , N+2))

    # Extend data and assign boundary conditions
    xe,qe[0,:] = extend(x,q[0,:], 1, "D", 1.0, "D", 0.125)
    xe,qe[1,:] = extend(x,q[1,:], 1, "D", 0.0, "N", 0.0)
    xe,qe[2,:] = extend(x,q[2,:], 1, "D", 2.5, "N", 0.0)

    # xe,qe[0,:] = extend(x,q[0,:], 1, "D", 3.857143, "N", 0.0)
    # xe,qe[1,:] = extend(x,q[1,:], 1, "D", 10.141852, "D", 0.0)
    # xe,qe[2,:] = extend(x,q[2,:], 1, "D", 39.166661, "N", 0.0)

    # Compute RHS - change numerical flux here

    dq = -1.0/h*( EulerLF(qe[:,1:N+1], qe[:,2:N+2], gamma, maxvel) - \
                   EulerLF(qe[:,0:N], qe[:,1:N+1], gamma, maxvel) ) 
    return dq

def EulerM1D(x,q,h,CFL,gamma,FinalTime):
    """Purpose: Integrate the 1D Euler equations until FinalTime 
                using a monotone scheme."""


    t = 0.0
    tstep = 0

    # Integrate scheme
    while t < FinalTime:
        # Set timestep
        p = (gamma-1)*(q[2,:] - 0.5*(q[1,:]**2)/q[0,:])
        c = np.sqrt(gamma*p/q[0,:])
        maxvel = (c+abs(q[1,:]/q[0,:])).max()
        k = min(FinalTime-t, CFL*h/maxvel)
        # Update solution
        q += k*EulerM1Drhs(x,q,gamma,h,k,maxvel)
        
        t = t+k
        tstep += 1
        
    return q

## 2D ##
###################################################################################
def EulerM2Drhs(x,y,q,gamma,hx,hy,k):
    """Purpose: Evaluate right hand side for the two-dimensional  
                Euler equations using monotone method""" 


    Ny, Nx = x.shape
    dq = np.zeros((4,Ny,Nx))

    # Apply monotone scheme in the x-direction
    qe = np.zeros((4, Nx+2))
    for i in range(Ny):
        # Extend data and assign boundary conditions in x-direction
        xe, qe[0,:] = extend(x[i,:], q[0,i,:], 1, "N", 0, "N", 0)
        xe, qe[1,:] = extend(x[i,:], q[1,i,:], 1, "N", 0, "N", 0)
        xe, qe[2,:] = extend(x[i,:], q[2,i,:], 1, "N", 0, "N", 0)
        xe, qe[3,:] = extend(x[i,:], q[3,i,:], 1, "N", 0, "N", 0)

        # Update residual
        dq[:,i,:] = -1.0/hx* (EulerLF2Dx(qe[:,1:Nx+1], qe[:,2:Nx+2], gamma) - \
                              EulerLF2Dx(qe[:,:Nx], qe[:,1:Nx+1], gamma) )

    
    #Apply monotone scheme in the y-direction   
    qe = np.zeros((4, Ny+2)) 
    for j in range(Nx):
        # Extend data and assign boundary conditions in y-direction
        ye, qe[0,:] = extend(y[:,j], q[0,:,j], 1, "N", 0, "N", 0)
        ye, qe[1,:] = extend(y[:,j], q[1,:,j], 1, "N", 0, "N", 0)
        ye, qe[2,:] = extend(y[:,j], q[2,:,j], 1, "N", 0, "N", 0)
        ye, qe[3,:] = extend(y[:,j], q[3,:,j], 1, "N", 0, "N", 0)

        # Update residual
        dq[:,:,j] -= 1.0/hy* (EulerLF2Dy(qe[:,1:Ny+1], qe[:,2:Ny+2], gamma) - \
                              EulerLF2Dy(qe[:,:Ny], qe[:,1:Ny+1], gamma) )
        
    return dq
    

def EulerM2D(x,y,q,hx,hy,gamma,CFL,FinalTime):
    """Purpose  : Integrate 2D Euler equation until FinalTime using 
                a monotone scheme.""" 

    t = 0.0
    tstep = 0
    delta = min(hx,hy)
    
    while t < FinalTime:
        # Set timestep
        p = (gamma-1)*(q[3,:,:] - 0.5*(q[1,:,:]**2+q[2,:,:]**2)/q[0,:,:])
        c = np.sqrt(gamma*p/q[0,:,:])
        maxvelx = (c+abs(q[1,:,:]/q[0,:,:])).max()
        maxvely = (c+abs(q[2,:,:]/q[0,:,:])).max()
        k = min(FinalTime-t, CFL*delta/(np.sqrt( 2*(maxvelx**2+maxvely**2) )))
        # Update solution
        q += k*EulerM2Drhs(x,y,q,gamma,hx,hy,k)
        
        t += k
        tstep += 1
        
    return q