import numpy as np

## 1D ##
###################################################################################
def EulerHLL(u,v,gamma):
    """Purpose: Compute flux for 1D Euler equations."""

    # Compute flux for u
    r = u[0,:]
    ru = u[1,:]
    E = u[2,:]
    pu = (gamma-1)*(E - 0.5*ru**2/r)

    cu = np.sqrt(gamma*pu/r)
    
    fu = np.array((ru, ru**2/r+pu, (E+pu)*ru/r))
    uu = ru/r
    sm = np.min( np.array((uu-cu, uu, uu+cu)), axis=0)

    # Compute flux for v
    r = v[0,:]
    ru = v[1,:]
    E = v[2,:]
    pv = (gamma-1)*(E - 0.5*ru**2/r)

    cv = np.sqrt(gamma*pv/r)

    fv = np.array((ru, ru**2/r+pv, (E+pv)*ru/r))
    uv = ru/r
    sp = np.max( np.array((uv-cv, uv, uv+cv)), axis=0)

    # Compute Roe average and extreme velocities
    us = (v*sp[None,:] - u*sm[None,:] + fu-fv)*((1.0/(sp-sm))[None,:])
    r = us[0,:]
    ru = us[1,:]
    E = us[2,:]
    ps = (gamma-1.0)*(E-0.5*ru**2/r)
    cs = np.sqrt(gamma*ps/r)
    uus = ru/r

    sm = np.min( np.array((sm, uus-cs, uus, uus+cs)), axis=0)
    sp = np.max( np.array((sp, uus-cs, uus, uus+cs)), axis=0)

    fs = (fu*sp[None,:]-fv*sm[None,:]+\
                (v-u)*((sp*sm)[None,:]))*((1.0/(sp-sm))[None,:])

    numflux = fu*((sm>0.0)[None,:]) + \
                fs*(((sm<=0.0)*(sp>=0.0))[None,:]) + \
                fv*((sp<0.0)[None,:])

    return numflux


def EulerHLLC(u,v,gamma):
    """Purpose: Compute flux for 1D Euler equations."""

    # Compute flux for u
    ru = u[0,:]
    ruu = u[1,:]
    Eu = u[2,:]
    pu = (gamma-1)*(Eu - 0.5*ruu**2/ru)

    cu = np.sqrt(gamma*pu/ru)
    
    fu = np.array((ruu, ruu**2/ru+pu, (Eu+pu)*ruu/ru))
    uu = ruu/ru
    sm = np.min( np.array((uu-cu, uu, uu+cu)), axis=0)

    # Compute flux for v
    rv = v[0,:]
    ruv = v[1,:]
    Ev = v[2,:]
    pv = (gamma-1)*(Ev - 0.5*ruv**2/rv)

    cv = np.sqrt(gamma*pv/rv)

    fv = np.array((ruv, ruv**2/rv+pv, (Ev+pv)*ruv/rv))
    uv = ruv/rv
    sp = np.max( np.array((uv-cv, uv, uv+cv)), axis=0)

    # Compute Roe pressure
    ss = (pv-pu+ruu*(sm-uu)-ruv*(sp-uv))/(ru*(sm-uu)-rv*(sp-uv))
    Plr = 0.5*(pu+pv+ru*(sm-uu)*(ss-uu) + rv*(sp-uv)*(ss-uv))

    # Compute numerical flux
    Ds = 0.0*u
    Ds[1,:] = 1.0
    Ds[2,:] = ss

    fus = ((u*sm[None,:]-fu)*ss[None,:] + \
            Ds*((Plr*sm)[None,:]))*((1.0/(sm-ss))[None,:])

    fvs = ((v*sp[None,:]-fv)*ss[None,:] + \
            Ds*((Plr*sp)[None,:]))*((1.0/(sp-ss))[None,:])


    numflux = fu*((sm>0.0)[None,:]) + \
                fus*(((sm<=0.0)*(ss>0.0))[None,:]) + \
                fvs*(((ss<=0.0)*(sp>0.0))[None,:]) + \
                fv*((sp<=0.0)[None,:])

    return numflux




## 2D ##
###################################################################################


def EulerHLL2Dx(ql,qr,gamma):
    """Purpose: Evaluate HLL numerical flux for Euler's equation along x."""

    # Extract states
    rl = ql[0,:]
    rul = ql[1,:]
    rvl = ql[2,:]
    El = ql[3,:]

    rr = qr[0,:]
    rur = qr[1,:]
    rvr = qr[2,:]
    Er = qr[3,:]

    # Compute fluxes for left and right states
    ul = rul/rl
    vl = rvl/rl
    pl = (gamma-1.0)*(El-0.5*rl*(ul**2+vl**2))
    cl = np.sqrt(gamma*pl/rl)

    Fl = np.array(( rul, rul*ul+pl, rul*vl, (El+pl)*ul ))
    sl = np.fmin( ul, np.fmin(ul+cl, ul-cl) )

    ur = rur/rr
    vr = rvr/rr
    pr = (gamma-1.0)*(Er-0.5*rr*(ur**2+vr**2))
    cr = np.sqrt(gamma*pr/rr)

    Fr = np.array(( rur, rur*ur+pr, rur*vr, (Er+pr)*ur ))
    sr = np.fmax( ur, np.fmax(ur+cr, ur-cr) ) 

    # Compute Roe average along x and velocity bounds
    qs = (qr*sr[None,:] - ql*sl[None,:] + Fl-Fr)*(((1.0)/(sr-sl))[None,:])

    us = qs[1,:]/qs[0,:]
    vs = qs[2,:]/qs[0,:]
    ps = (gamma-1.0)*(qs[3,:] - 0.5*qs[0,:]*(us**2+vs**2))
    cs = np.sqrt(gamma*ps/qs[0,:])

    ssmi = np.fmin( us, np.fmin(us+cs,us-cs))
    ssma = np.fmax( us, np.fmax(us+cs,us-cs))

    sm = np.fmin(sl,ssmi)
    sp = np.fmax(sr,ssma)

    # Compute  flux
    dq = np.zeros(ql.shape)
    q1 = sm > 0.0
    q2 = (sp >= 0.0)*(sm <= 0.0)
    q3 = sp < 0.0

    fs = (Fl*sp[None,:] - Fr*sm[None,:] + \
             (qr-ql)*((sm*sp)[None,:]))*(((1.0)/(sp-sm))[None,:])

    dq = Fl*q1[None,:] + fs*q2[None,:] + Fr*q3[None,:]

    return dq

def EulerHLL2Dy(ql,qr,gamma):
    """Purpose: Evaluate HLL numerical flux for Euler's equation along y."""

    # Extract states
    rl = ql[0,:]
    rul = ql[1,:]
    rvl = ql[2,:]
    El = ql[3,:]

    rr = qr[0,:]
    rur = qr[1,:]
    rvr = qr[2,:]
    Er = qr[3,:]

    # Compute fluxes for left and right states
    ul = rul/rl
    vl = rvl/rl
    pl = (gamma-1.0)*(El-0.5*rl*(ul**2+vl**2))
    cl = np.sqrt(gamma*pl/rl)

    Fl = np.array(( rvl, rvl*ul, rvl*vl+pl, (El+pl)*vl ))
    sl = np.fmin( vl, np.fmin(vl+cl, vl-cl) )

    ur = rur/rr
    vr = rvr/rr
    pr = (gamma-1.0)*(Er-0.5*rr*(ur**2+vr**2))
    cr = np.sqrt(gamma*pr/rr)

    Fr = np.array(( rvr, rvr*ur, rvr*vr+pr, (Er+pr)*vr ))
    sr = np.fmax( vr, np.fmax(vr+cr, vr-cr) ) 

    # Compute Roe average along y and velocity bounds
    qs = (qr*sr[None,:] - ql*sl[None,:] + Fl-Fr)*(((1.0)/(sr-sl))[None,:])

    us = qs[1,:]/qs[0,:]
    vs = qs[2,:]/qs[0,:]
    ps = (gamma-1.0)*(qs[3,:] - 0.5*qs[0,:]*(us**2+vs**2))
    cs = np.sqrt(gamma*ps/qs[0,:])

    ssmi = np.fmin( us, np.fmin(vs+cs,vs-cs))
    ssma = np.fmax( us, np.fmax(vs+cs,vs-cs))

    sm = np.fmin(sl,ssmi)
    sp = np.fmax(sr,ssma)

    # Compute  flux
    dq = np.zeros(ql.shape)
    q1 = sm > 0.0
    q2 = (sp >= 0.0)*(sm <= 0.0)
    q3 = sp < 0.0

    fs = (Fl*sp[None,:] - Fr*sm[None,:] + \
             (qr-ql)*((sm*sp)[None,:]))*(((1.0)/(sp-sm))[None,:])

    dq = Fl*q1[None,:] + fs*q2[None,:] + Fr*q3[None,:]

    return dq