"""
This file contains the specific functions to solve Euler equations
in 1D using a flux-limited scheme.
"""

import numpy as np
from Euler import *
from helpers import extend,FluxLimit

## 1D ##
###################################################################################
def EulerFLrhs1D(x,q,gamma,h,k,maxvel):

    """Purpose: Evaluate right hand side for Euler equation using a
                flux limited scheme"""
    N = len(x)

    # Chose flux limiter - 0:LF; 1:CO; 2:Koren; 3:Sweby; 4:OSPRE; 5:van Leer
    typ = 5
    beta = 1.0

    # Boundary conditions 
    xe, re = extend(x, q[0,:], 1, "D", 1.0, "D", 0.125)
    xe, me = extend(x, q[1,:], 1, "D", 0.0, "N", 0.0)
    xe, Ee = extend(x, q[2,:], 1, "D", 2.5, "N", 0.0)

    # xe, re = extend(x, q[0,:], 1, "D", 3.857143, "N", 0.0)
    # xe, me = extend(x, q[1,:], 1, "D", 10.141852, "D", 0.0)
    # xe, Ee = extend(x, q[2,:], 1, "D", 39.166661, "N", 0.0)

    # Compute indicator function and define flux limiter
    rr = (re[1:N+1] - re[:N])/(re[2:N+2] - re[1:N+1])
    rm = (me[1:N+1] - me[:N])/(me[2:N+2] - me[1:N+1])
    rE = (Ee[1:N+1] - Ee[:N])/(Ee[2:N+2] - Ee[1:N+1])

    xe,rre = extend(x,rr,1,"N",0,"N",0)
    xe,rme = extend(x,rm,1,"N",0,"N",0)
    xe,rEe = extend(x,rE,1,"N",0,"N",0)

    phirL = FluxLimit( rre[:N], typ, beta )
    phirR = FluxLimit( rre[1:N+1], typ, beta)
    phimL = FluxLimit( rme[:N], typ, beta)
    phimR = FluxLimit( rme[1:N+1], typ, beta)
    phiEL = FluxLimit( rEe[:N], typ, beta)
    phiER = FluxLimit( rEe[1:N+1], typ, beta)

    phiL = np.array([phirL, phimL, phiEL])
    phiR = np.array([phirR, phimR, phiER])

    # Compute fluxes
    q = np.array([re[1:N+1],me[1:N+1],Ee[1:N+1]])
    qp = np.array([re[2:N+2],me[2:N+2],Ee[2:N+2]])
    qm = np.array([re[:N],me[:N],Ee[:N]])

    FluxlowR = EulerLF(q, qp, gamma, maxvel)
    FluxhighR = EulerLW(q, qp, gamma, k/h)
    FluxlowL = EulerLF(qm, q, gamma, maxvel)
    FluxhighL = EulerLW(qm, q, gamma, k/h)

    FluxL = FluxlowL + phiL*(FluxhighL-FluxlowL)
    FluxR = FluxlowR + phiR*(FluxhighR-FluxlowR)


    # Compute RHS
    du = - (FluxR-FluxL)/h

    return du

def EulerFLcharrhs1D(x,q,gamma,h,k,maxvel):

    """Purpose: Evaluate right hand side for Euler equation using a
                flux limited scheme"""
    N = len(x)
    phir = np.zeros(3)
    dq = np.zeros((3,N))

    # Chose flux limiter - 0:LF; 1:CO; 2:Koren; 3:Sweby; 4:OSPRE; 5:van Leer
    typ = 5
    beta = 1.5

    # Boundary conditions 
    xe, re = extend(x, q[0,:], 2, "D", 1.0, "D", 0.125)
    xe, me = extend(x, q[1,:], 2, "D", 0.0, "N", 0.0)
    xe, Ee = extend(x, q[2,:], 2, "D", 2.5, "N", 0.0)

    # xe, re = extend(x, q[0,:], 1, "D", 3.857143, "N", 0.0)
    # xe, me = extend(x, q[1,:], 1, "D", 10.141852, "D", 0.0)
    # xe, Ee = extend(x, q[2,:], 1, "D", 39.166661, "N", 0.0)

    qe = np.array((re,me,Ee))
    for i in range(N):
        qloc = qe[:,i:i+5]

        # Left cell interface
        q0 = 0.5*(qloc[:,1]+qloc[:,2])
        S,iS,Lam = EulerChar(q0,gamma)
        Rloc = np.dot(iS,qloc)
        for j in range(3):
            if Lam[j,j]>=0.0: # Upwind
                phir[j] = FluxLimit( (Rloc[j,1]-Rloc[j,0])/(Rloc[j,2]-Rloc[j,1]), \
                                    typ, beta )
            else: # Downwind
                phir[j] = FluxLimit( (Rloc[j,3]-Rloc[j,2])/(Rloc[j,2]-Rloc[j,1]), \
                                    typ, beta )
        phiL = np.dot(S, np.dot(np.diag(phir),iS))

        # Right cell interface
        q0 = 0.5*(qloc[:,2]+qloc[:,3])
        S,iS,Lam = EulerChar(q0,gamma)
        Rloc = np.dot(iS,qloc)
        for j in range(3):
            if Lam[j,j]>=0.0: # Upwind
                phir[j] = FluxLimit( (Rloc[j,2]-Rloc[j,1])/(Rloc[j,3]-Rloc[j,2]), \
                                    typ, beta )
            else: # Downwind
                phir[j] = FluxLimit( (Rloc[j,4]-Rloc[j,3])/(Rloc[j,3]-Rloc[j,2]), \
                                    typ, beta )
        phiR = np.dot(S, np.dot(np.diag(phir),iS))

        # Compute fluxes
        q = np.array([re[i+2],me[i+2],Ee[i+2]])
        qp = np.array([re[i+3],me[i+3],Ee[i+3]])
        qm = np.array([re[i+1],me[i+1],Ee[i+1]])

        FluxlowR = EulerLF(np.transpose(np.array([q])),\
                           np.transpose(np.array([qp])), gamma, maxvel)
        FluxhighR = EulerLW(np.transpose(np.array([q])),\
                           np.transpose(np.array([qp])), gamma, k/h)
        FluxlowL = EulerLF(np.transpose(np.array([qm])),\
                           np.transpose(np.array([q])), gamma, maxvel)
        FluxhighL = EulerLW(np.transpose(np.array([qm])),\
                           np.transpose(np.array([q])), gamma, k/h)

        FluxL = FluxlowL + np.dot(phiL,(FluxhighL-FluxlowL))
        FluxR = FluxlowR + np.dot(phiR,(FluxhighR-FluxlowR))


        # Compute RHS
        dq[:,i] = - (FluxR-FluxL)[:,0]/h

    return dq

def EulerFL1D(x,q,h,CFL,gamma,FinalTime):
    """Purpose  : Integrate 1D Euler equation until FinalTime using a flux limited scheme.
    """   
    t = 0.0
    timestep = 0

    while t < FinalTime:
        # Set timestep
        p = (gamma-1.0)*(q[2,:] - 0.5*q[1,:]**2/q[0,:])
        c = np.sqrt(gamma*p/q[0,:])
        maxvel = (c + np.abs(q[1,:]/q[0,:])).max()
        k = min(FinalTime-t, CFL*h/maxvel)
        q += k*EulerFLrhs1D(x,q,gamma,h,k,maxvel)
        t +=k
        timestep += 1
        
    return q
