import numpy as np
from Legendre import *
from helpers import extendDG,minmod


def WENODGWeights(m,iV):
	"""Purpose: Compute operators to enable evaluation of WENO smoothness
				indicator and WENO polynomial of order m."""

	Q = np.zeros((m+1,m+1))
	Pmat = np.zeros((m+1,m+1))
	Xm = np.zeros((m+1,m+1))
	Xp = np.zeros((m+1,m+1))

	# Compute quadrature points
	x,w = LegendreGQ(m)
	Lambda = np.diag(w)

	# Initial matrices of Legendre polynomials
	for i in range(m+1):
		Pmat[i,:] = LegendreP(x,i)
		Xm[i,:] = LegendreP(x-2.0,i)
		Xp[i,:] = LegendreP(x+2.0,i)

	# Compute matrices corresponding to increasing order of derivative
	for l in range(1,m+1):
		A = np.zeros((m+2-l,m+2-l))
		A[0,0] = 1.0/np.sqrt((2*l+1)*(2*l-1))
		A[-1,-1] = 1.0/(np.sqrt( 2*(m+2)+1 )*np.sqrt( 2*(m+2)-1 ))
		for i in range(2,m-l+2):
			Ah = 1.0/(np.sqrt(2*(l-1+i)+1)*np.sqrt(2*(l-1+i)-1))
			A[i-1,i-1] = Ah
			A[i,i-2] = -Ah

		# Recover derivatives at quadrature points
		Ph1 = np.dot(np.linalg.inv(A),Pmat[l-1:m+1,:])

		Pmat[:l,:] = 0.0
		Pmat[l:m+1,:] = Ph1[:m-l+1,:]

		# Compute smoothness operator for order l and update
		Qh = np.dot(Pmat, np.dot(Lambda, np.transpose(Pmat)))
		Q += 2**(2*l-1)*Qh

	# Initialize operator for smoothness indicator in nodal space
	Q = np.dot(np.transpose(iV), np.dot(Q, iV))

	# Initialize interpolation matrices
	Xp = np.dot(np.transpose(iV), Xp)
	Xm = np.dot(np.transpose(iV), Xm)

	return Q, Xm, Xp

def VandermondeDG(m,r):
	"""Purpose : Initialize the 1D Vandermonde Matrix, V_{ij} = phi_j(r_i);"""

	V = np.zeros((len(r),m+1))
	for i in range(m+1):
		V[:,i] = LegendreP(r,i)

	return V

def GradVandermondeDG(m,r):
	"""Purpose : Initialize the gradient of the Vandermonde matrix 
				 of order m at (r)"""

	Vr = np.zeros((len(r),m+1))	
	for i in range(m+1):
		Vr[:,i] = GradLegendreP(r,i)

	return Vr

def DmatrixDG(m,r,V):
	"""Purpose : Initialize the (r) differentiation matrices, 
				evaluated at (r) at order m"""

	Vr = GradVandermondeDG(m,r)
	D = np.dot(Vr,np.linalg.inv(V))

	return D


def WENOlimitDG(x,u,m,h,N,V,iV,Q,Xm,Xp):
	"""Purpose: Apply WENO limiter by Zhong-Shu (2013) 
				to u - an m'th order polynomial"""

	eps0 = 1.0e-6

	# Set constants for limiting
	eps1 = 1.0e-10
	p = 1.0

	gammam1 = 0.001
	gamma0 = 0.998
	gammap1 = 0.001

	# Compute cell averages and cell centers
	uh = np.dot(iV,u)

	uh[1:m+1,:] = 0
	uavg = np.dot(V,uh)
	ucell = uavg[0,:]
	ulimit = np.copy(u)

	# Compute extended polynomials with zero cell averages
	ue = extendDG(u,"P",0.0,"P",0.0)

	Pm = np.dot(np.transpose(Xp),ue)

	Pp = np.dot(np.transpose(Xm),ue)
	Ph = np.dot(iV,Pm)
	Ph[0,:] = 0.0
	Pm = np.dot(V,Ph)
	Ph = np.dot(iV,Pp)
	Ph[0,:] = 0.0
	Pp = np.dot(V,Ph)
	

	#Extend cell averages
	ve = extendDG(ucell, "P", 0.0, "P", 0.0)

	# Extract end values and cell averages for each element
	uel = u[0,:]
	uer = u[-1,:]
	vj = ucell
	vjm = ve[:N]
	vjp = ve[2:N+2]

	# Find elements that require limiting
	vel = vj - minmod( np.array((vj-uel, vj-vjm, vjp-vj )) )
	ver = vj + minmod( np.array((uer-vj, vj-vjm, vjp-vj )) )

	ids = np.where(np.logical_or(np.abs(vel-uel)>eps0,np.abs(ver-uer)>eps0))


	# Apply limiting when needed
	if ids[0].size:
		# Extract local polynomials
		pm1 = Pm[:,ids[0]] + np.outer(np.ones(m+1),vj[ids[0]])
		p0 = u[:,ids[0]]
		pp1 = Pp[:,ids[0]+2] + np.outer(np.ones(m+1),vj[ids[0]])


		# Compute smoothness indicators and WENO weights

		betam1 = np.diag(np.dot(np.transpose(pm1),np.dot(Q,pm1)))
		alpham1 = gammam1/(eps1+betam1)**(2*p)
		beta0 = np.diag(np.dot(np.transpose(p0),np.dot(Q,p0)))
		alpha0 = gamma0/(eps1+beta0)**(2*p)
		betap1 = np.diag(np.dot(np.transpose(pp1),np.dot(Q,pp1)))
		alphap1 = gammap1/(eps1+betap1)**(2*p)

		alphas = alpham1 + alpha0 + alphap1
		omm1 = alpham1/alphas
		om0 = alpha0/alphas
		omp1 = alphap1/alphas

		# Compute limited function
		ulimit[:,ids[0]] = np.dot(pm1,np.diag(omm1)) + \
						np.dot(p0,np.diag(om0)) +  \
						np.dot(pp1,np.diag(omp1))

	return ulimit


def SlopeLimitCSDG(x,u,m,h,N,V,iV):
	"""Purpose: Apply slopelimiter by Cockburn-Shu (1989) 
				to u - an m'th order polynomial  """

	eps0 = 1.0e-8

	# Strength of slope limiter - Minmod: theta=1, MUSCL:theta=2
	theta = 2.0

	# Compute cell averages and cell centers
	uh = np.dot(iV,u)

	uh[1:m+1,:] = 0
	uavg = np.dot(V,uh)
	ucell = uavg[0,:]
	ulimit = np.copy(u)
	
	#Extend cell averages
	ve = extendDG(ucell, "P", 0.0, "P", 0.0)

	# Extract end values and cell averages for each element
	uel = u[0,:]
	uer = u[-1,:]
	vj = ucell
	vjm = ve[:N]
	vjp = ve[2:N+2]

	# Find elements that require limiting
	vel = vj - minmod( np.array((vj-uel, vj-vjm, vjp-vj )) )
	ver = vj + minmod( np.array((uer-vj, vj-vjm, vjp-vj )) )

	ids = np.where(np.logical_or(np.abs(vel-uel)>eps0,np.abs(ver-uer)>eps0))


	# Apply limiting when needed
	if ids[0].size:
		# Create piecewise linear solution for limiting on specified elements
		uhl = np.dot(iV,u[:,ids[0]])
		uhl[2:m+1,:] = 0.0
		ulin = np.dot(V,uhl)
		ux = 2.0/h*(vj[ids[0]]-ulin[0,:])

		# Limit function
		x0h = 0.5*np.outer(np.ones(m+1), x[-1,:]+x[0,:] )

		v = np.array((ux, \
						theta/h*(vjp[ids[0]]-vj[ids[0]]), \
						theta/h*(vj[ids[0]]-vjm[ids[0]]), ))

		ulimit[:,ids[0]] = np.outer( np.ones(m+1), vj[ids[0]]) + \
							(x[:,ids[0]]-x0h[:,ids[0]])* \
							np.outer( np.ones(m+1), minmod(v))

	return ulimit




def MomentLimitDG(x,u,m,h,N,V,iV):
	"""Purpose: Apply moment limiter 
				to u - an m'th order polynomial"""

	eps0 = 1.0e-8
	eps1 = 1.0e-8

	# Strength of slope limiter - Minmod: theta=1, MUSCL:theta=2
	theta = 2.0

	# Compute cell averages and cell centers
	uh = np.dot(iV,u)

	uh[1:m+1,:] = 0
	uavg = np.dot(V,uh)
	ucell = uavg[0,:]
	ulimit = np.copy(u)
	
	#Extend cell averages
	ve = extendDG(ucell, "N", 0.0, "N", 0.0)

	# Extract end values and cell averages for each element
	uel = u[0,:]
	uer = u[-1,:]
	vj = ucell
	vjm = ve[:N]
	vjp = ve[2:N+2]

	# Find elements that require limiting
	vel = vj - minmod( np.array((vj-uel, vj-vjm, vjp-vj )) )
	ver = vj + minmod( np.array((uer-vj, vj-vjm, vjp-vj )) )

	ids = (np.logical_and(np.abs(vel-uel)<eps1,np.abs(ver-uer)<eps1))
	mark = np.zeros(N)
	mark = np.logical_or( ids, mark)

	# Compute expansion coefficients
	uh = np.dot(iV, u)


	# Apply limiting when needed
	for i in range(m+1,0,-1):
		# Create piecewise linear solution for limiting on specified elements
		uh1 = uh[i-1,:]
		uh2 = uh[i-2,:]
		uh2e = extendDG(uh2,"P",0,"P",0)
		uh2m = uh2e[:N]
		uh2p = uh2e[2:N+2]

		con = np.sqrt( (2*i+1)*(2*i-1) )

		uh1 = uh1*mark + (1.0 - mark)/con* \
						minmod( np.array( ( con*uh1, theta*(uh2p-uh2),\
								 theta*(uh2-uh2m)) ) )

		idsh = np.abs(uh1-uh[i-1,:])<eps0
		mark = np.logical_or(idsh, mark)

	ulimit = np.dot(V,uh)

	return ulimit



def SlopeLimitBSBDG(x,u,m,h,N,V,iV):
	"""Purpose: Apply slopelimiter by Burbeau-Sagaut-Bruneau (2001)
				to u - an m'th order polynomial  """

	eps0 = 1.0e-8

	# Strength of slope limiter - Minmod: theta=1, MUSCL:theta=2
	theta = 2.0

	# Compute cell averages and cell centers
	uh = np.dot(iV,u)
	uhx = np.copy(uh)
	uh[1:m+1,:] = 0.0
	uavg = np.dot(V,uh)
	ucell = uavg[0,:]
	uhx[2:m+1,:] = 0.0
	ulin = np.dot(V,uhx)
	ux = 2.0/h*(ucell-ulin[0,:])
	ulimit = np.copy(u)
	
	#Extend cell averages
	ve = extendDG(ucell, "P", 0.0, "P", 0.0)
	vxe = extendDG(ux, "P", 0.0, "P", 0.0)

	# Extract end values and cell averages for each element
	uel = u[0,:]
	uer = u[-1,:]
	vj = np.copy(ucell)
	vjm = ve[:N]
	vjp = ve[2:N+2]
	vxj = np.copy(ux)
	vxjm = vxe[:N]
	vxjp = vxe[2:N+2]

	# Find elements that require limiting
	vel = vj - minmod( np.array((vj-uel, vj-vjm, vjp-vj )) )
	ver = vj + minmod( np.array((uer-vj, vj-vjm, vjp-vj )) )

	ids = np.where(np.logical_or(np.abs(vel-uel)>eps0,np.abs(ver-uer)>eps0))


	# Apply limiting when needed
	if ids[0].size:
		# Create piecewise linear solution for limiting on specified elements
		uhl = np.dot(iV,u[:,ids[0]])
		uhl[2:m+1,:] = 0.0
		ulin = np.dot(V,uhl)

		# Limit function
		x0h = 0.5*np.outer(np.ones(m+1), x[-1,:]+x[0,:] )

		ux1 = minmod( np.array(( vxj[ids[0]] , \
								theta*(vjp[ids[0]]-vj[ids[0]])/h , \
								theta*(vj[ids[0]]-vjm[ids[0]])/h )) )
		ux2 = minmod( np.array(( vxj[ids[0]], vxjm[ids[0]], vxjp[ids[0]] )))

		v = np.array((ux1,ux2 ))

		ulimit[:,ids[0]] = np.outer( np.ones(m+1), vj[ids[0]]) + \
							(x[:,ids[0]]-x0h[:,ids[0]])* \
							np.outer( np.ones(m+1), maxmod(v))

	return ulimit


def extpreimitDG(x,u,m,h,N,V,iV,umin,umax):
	"""Purpose: Apply extrema preserving limiter to piecewise
		polynomial solution u - an m'th order polynomial  """

	eps0 = 1.0e-8

	# Compute cell averages and cell centers
	uh = np.dot(iV,u)

	uh[1:m+1,:] = 0
	uavg = np.dot(V,uh)
	ucell = uavg[0,:]
	ulimit = np.copy(u)
	
	#Extend cell averages
	ve = extendDG(ucell, "P", 0.0, "P", 0.0)

	# Extract end values and cell averages for each element
	uel = u[0,:]
	uer = u[-1,:]
	vj = ucell
	vjm = ve[:N]
	vjp = ve[2:N+2]

	# Find elements that require limiting
	vel = vj - minmod( np.array((vj-uel, vj-vjm, vjp-vj )) )
	ver = vj + minmod( np.array((uer-vj, vj-vjm, vjp-vj )) )

	ids = np.where(np.logical_or(np.abs(vel-uel)>eps0,np.abs(ver-uer)>eps0))


	# Apply limiting when needed
	if ids[0].size:

		minu = np.min(u[:,ids[0]],axis=0)
		maxu = np.max(u[:,ids[0]],axis=0)
		ulimmax = np.abs((umax-ucell[ids[0]])/(maxu-ucell[ids[0]]))
		ulimmin = np.abs((umin-ucell[ids[0]])/(minu-ucell[ids[0]]))
		theta = np.min( np.array((ulimmax, ulimmin, np.ones(len(ids[0])))), axis=0 )

		ulimit[:,ids[0]] = uavg[:,ids[0]] + \
							(u[:,ids[0]] - uavg[:,ids[0]])*\
							 np.outer( np.ones(m+1), theta)

	return ulimit



def FilterDG(m,mc,p,V):
	"""Purpose : Initialize DG filter matrix of size m. 
		Order of exponential filter is (even) p with cutoff at mc"""

	filterdiag = np.ones(m+1)
	alpha = -np.log(np.finfo(float).eps)

	# Initialize filter function
	for i in range(mc,m+1):
		filterdiag[i] = np.exp(-alpha*((i-mc)/(m-mc))**p)

	F = np.dot(V, np.dot( np.diag(filterdiag), np.linalg.inv(V)))
	return F
