"""
This file contains the specific functions to solve Burgers equations
in 1D or 2D using a monotone scheme.
"""

import numpy as np
from Burgers import *
from helpers import extend

## 1D ##
###################################################################################
def BurgersMrhs1D(x, u, h, k, maxvel):

    """Purpose: Evaluate right hand side for Burgers equation 
                using a monotone scheme"""
    N = len(x)

    xe,ue = extend(x, u, 1, "P", 0, "P", 0) # Periodic boundary conditions
    # xe,ue = extend(x, u, 1, "D", 2, "N", 0) # Constant boundary conditions

    # Change numerical flux here
    du = - (BurgersLF(ue[1:N+1], ue[2:N+2], maxvel) - \
            BurgersLF(ue[:N], ue[1:N+1], maxvel))/h
    return du

def BurgersM1D(x, u, h, CFL, FinalTime):
    """Purpose: Integrate 1D Burgers equation until FinalTime 
                using a monotone scheme.
    """   
    t = 0.0
    timestep = 0

    while t < FinalTime:
        # Decide on timestep
        maxvel = (2.0*np.abs(u)).max()
        k = min(FinalTime-t, CFL*h/maxvel)
        # Update solution
        u += k*BurgersMrhs1D(x,u,h,k,maxvel)
        t +=k
        timestep += 1
        
    return u

## 2D ##
###################################################################################
def BurgersMrhs2D(x, y, u, hx, hy, k, maxvel):
    """Purpose: Evaluate right hand side for 2D Burgers equation 
                using a monotone scheme
    """   

    Ny,Nx = u.shape

    du = np.zeros((Ny,Nx))

    # Extend data and assign boundary conditions in x-direction
    for i in range(Ny):
        xe,ue = extend(x[i,:], u[i,:], 1, "P", 0, "P", 0)
        # Update residual
        du[i,:] = -1.0/hx* (BurgersLF(ue[1:-1], ue[2:], maxvel) - \
                              BurgersLF(ue[0:-2], ue[1:-1], maxvel) )

    # Extend data and assign boundary conditions in y-direction
    for j in range(Nx):
        ye, ue = extend(y[:,j], u[:,j], 1, "P", 0, "P", 0)
        # Update residual
        du[:,j] -= 1.0/hy* (BurgersLF(ue[1:-1], ue[2:], maxvel) - \
                              BurgersLF(ue[0:-2], ue[1:-1], maxvel) )
        
    return du
    

def BurgersM2D(x, y, u, hx, hy, CFL, FinalTime):
    """Purpose: Integrate 2D Burgers equation until FinalTime 
                using a monotone scheme. 
    """   
    t = 0.0
    tstep = 0
    delta = min(hx,hy)
    
    while t < FinalTime:
        # Decide on timestep
        maxvel = (2*np.sqrt(2.0)*np.abs(u)).max()
        k = min(FinalTime-t, CFL*delta/maxvel/2)
        # Update solution
        u += k*BurgersMrhs2D(x, y, u, hx, hy, k, maxvel)
        t += k
        tstep += 1
        
    return u