"""
This file contains the specific functions to solve Burgers equations
in 1D using a discontinuous galerkin scheme.
"""

import numpy as np
from Burgers import *
from DiscontinuousGalerkin import *
from helpers import extendDG

## 1D ##
###################################################################################
def BurgersDGrhs1D(x,u,h,k,m,N,Ma,S,VtoE,maxvel):
    """Purpose  : Evaluate the RHS of Burgers equations using a DG method"""
    Imat = np.eye(m+1)

    # Extend data and assign boundary conditions
    ue = extendDG(u[VtoE[0],VtoE[1]], "P", 0, "P", 0) 

    # Compute numerical fluxes at interfaces
    fluxr = BurgersLF( ue[1,1:N+1], ue[0,2:N+2], maxvel)
    fluxl = BurgersLF( ue[1,:N], ue[0,1:N+1], maxvel)

    # Compute right hand side of Burgers equation
    ru = np.dot(np.transpose(S), u**2) - (np.outer(Imat[:,m], fluxr) - \
                np.outer(Imat[:,0], fluxl))

    rhsu = np.dot( np.linalg.inv(h/2*Ma), ru )
    return rhsu

def BurgersDG1D(x,u,h,m,N,CFL,FinalTime):
    """Purpose  : Integrate 1D Burgers equation until FinalTime using a DG
                  scheme and 3rd order SSP-RK method
    """   

    # Initialize operators at Legendre Gauss Lobatto grid
    r,w = LegendreGL(m)
    V = VandermondeDG(m,r)
    D = DmatrixDG(m, r, V)

    Ma = np.linalg.inv(np.dot(V,np.transpose(V)))
    S = np.dot(Ma,D)
    iV = np.linalg.inv(V)

    # Compute operator for WENO smoothness evaluator
    Q,Xm,Xp = WENODGWeights(m,iV)

    # Initialize extraction vector
    VtoE = [np.array(([0]*N, [m]*N)) , np.arange(N)]

    # # Initialize filter matrix
    # F = FilterDG(m,0,10,V)

    # Compute smallest spatial scale timestep
    rLGLmin = np.abs(r[0]-r[1]).min()

    t = 0.0
    tstep = 0

    # Initialize parameters for nonlinear viscosity
    nu = np.zeros((m+1,N))
    nu0 = 2.0
    kappa = -6.0
    c2 = 1.0

    # Integrate scheme
    while (t<FinalTime):
        # Decide on timestep
        maxvel = (2*np.abs(u)).max()
        k = min(CFL*rLGLmin*h/maxvel,FinalTime-t)
        # Update solution - stage 1
        rhsu  = BurgersDGrhs1D(x,u,h,k,m,N,Ma,S,VtoE,maxvel)
        u1 = u + k*rhsu
        u1 = WENOlimitDG(x,u1,m,h,N,V,iV,Q,Xm,Xp)
        # Update solution - stage 2
        rhsu  = BurgersDGrhs1D(x,u1,h,k,m,N,Ma,S,VtoE,maxvel) 
        u2 = (3*u + u1 + k*rhsu)/4
        u2 = WENOlimitDG(x,u2,m,h,N,V,iV,Q,Xm,Xp)
        # Update solution - stage 3
        rhsu  = BurgersDGrhs1D(x,u2,h,k,m,N,Ma,S,VtoE,maxvel)
        u = (u + 2*u2 + 2*k*rhsu)/3
        u = WENOlimitDG(x,u,m,h,N,V,iV,Q,Xm,Xp)
        
        t += k
        tstep += 1

    return u
