"""
This file contains all the basic implementations of functions to compute
quantities for the specific case of the Burgers equations.
"""

import numpy as np

## 1D ##
###################################################################################
def BurgersFlux(u):
    """Purpose: Evaluate flux for Burgers"""

    return u**2

def BurgersJac(u):
    """Purpose: Evaluate Jacobian for Burgers flux"""

    return 2*u


def BurgersLF(u, v, maxvel):
    """Purpose: Evaluate the Lax Friedrich numerical flux for Burgers equation"""
    
    fu = u**2
    fv = v**2
    
    flux = 0.5*(fu+fv)-0.5*maxvel*(v-u)
    return flux

def BurgersLW(u, v, lambd):
	"""Purpose: Evaluate the Lax Wendroff numerical flux for Burgers equation"""

	fu = u**2
	fv = v**2

	alpha = lambd*(u+v)

	numflux = 0.5*(fu+fv) - 0.5*alpha*(fv-fu)
	return numflux

def BurgersRoe(u,v):
	"""Purpose: Purpose: Evaluate Roe numerical flux for Burgers equation.
				No sonic fix."""

	fu = u**2
	fv = v**2
	alpha = u+v

	numflux = (alpha >= 0)*fu + (alpha < 0)*fv
	return numflux

## 2D ##
###################################################################################
def BurgersFlux2Dx(u):
    """Purpose: Evaluate x-component of flux for 2D Burgers"""

    return u**2

def BurgersFlux2Dy(u):
    """Purpose: Evaluate y-component of flux for 2D Burgers"""

    return u**2

def BurgersJac2Dx(u):
    """Purpose: Evaluate x-component of flux for Jacobian 2D Burgers"""

    return 2*u


def BurgersJac2Dy(u):
    """Purpose: Evaluate y-component of flux for Jacobian 2D Burgers"""

    return 2*u