function [sd] = Test_single_StableImplementation_Nystrom(n,r,R,q,field,model)
%This script runs an experimental instance using a matrix with exponential 
%decay in the spectrum. This test underlines the importance of the stable
%implementation in terms of the numerical errors. See Section B in 
%Supplements of [TYUC2017Nys] for details.
%	Inputs: n 		-> size of input matrix (n x n)
%			r		-> target rank
%			R		-> Rank of "significant part"
%			q		-> parameter that controls the rate of decay
% 			field 	-> field for the test matrices: 'complex' or 'real'.
%           model   -> model for the test matrices: 'orthonormal' or 'ssft'
%	Output: sd		-> structure with following fields
%						* ErrBest_S2 ,ErrBest_S1, ErrBest_Sinf
%						* info, k, l, ID
%						* ErrSta_S2, ErrSta_S1, ErrSta_Sinf
%						* ErrUns_S2, ErrUns_S1, ErrUns_Sinf
%
%   [LLSSKT2017] H. Li, G.C. Linderman, A.Szlam, K.P. Stanton, Y. Kluger 
%   and M. Tygert. Algorithm 971: An Implementation of a Randomized 
%   Algorithm for Principal Component Analysis. ACM Trans. Math. Softw., 
%   43 (3):28:1?28:14, Jan. 2017.
%
%   [TYUC2017Nys] J.A. Tropp, A. Yurtsever, M. Udell and V. Cevher. Fixed-
%   Rank Approximation of a Positive-Semidefinite Matrix from Streaming 
%   Data. In Proc. 31st Conference on Neural Information Processing Systems
%   (NIPS), Long Beach, CA, USA, December 2017.
%
%   Coded by: Alp Yurtsever
%   Ecole Polytechnique Federale de Lausanne, Switzerland.
%   Laboratory for Information and Inference Systems, LIONS.
%   contact: alp.yurtsever@epfl.ch
%   Created: May 13, 2017
%   Last modified: October 23, 2017
%
%Nys-SKETCHv1.0
%Copyright (C) 2017 Laboratory for Information and Inference Systems
%(LIONS), Ecole Polytechnique Federale de Lausanne, Switzerland.
%This code is a part of Nys-SKETCH toolbox. 
%Please read COPYRIGHT before using this file.


%% Create or load data

% Generate the matrix A
Sigma(1:R) = 1;
Sigma((R+1):n) = 10.^(-q*((1:(n-R))+1));
A = diag(Sigma+0*1i);
% Find the ground truth (best rank-r approximation)
ErrBest_S2 = norm(Sigma(r+1:end));
ErrBest_S1 = norm(Sigma(r+1:end),1);
ErrBest_Sinf = norm(Sigma(r+1:end),inf);

%% Beginning of the test
T = unique([r:2:2*r, 2*r, round((2*r+3) * 2.^(0:.5:4))]);

for Iter = 1:numel(T)
    
    %% NYSTROM Sketch implemented as in [TYUC2017Nys,Algorithm(3)]
    myNystromSketch = NystromSketch(A, T(Iter), field, model);
    Astable = myNystromSketch.FixedRankPSDApprox(r);
    ErrSta_S2 = norm(Astable - A,'fro');
    Z = eig(Astable - A,'vector');
    ErrSta_S1 = norm(Z,1);
    ErrSta_Sinf = norm(Z,inf);
    
    %% NYSTROM Sketch implemented as in [LLSSKT2017, Eqn.(13)].
    Aunstable = myNystromSketch.FixedRankPSDApproxUnstable(r);
    ErrUns_S2  = norm(Aunstable - A,'fro');
    Z = eig(Aunstable - A,'vector');
    ErrUns_S1 = norm(Z,1);
    ErrUns_Sinf = norm(Z,inf);
    
    %% Variables to be saved
    sd.k{Iter} = [];
    sd.l{Iter} = [];
    sd.t{Iter} = [];
    sd.v{Iter} = [];
    sd.ErrSta_S2{Iter} = ErrSta_S2;
    sd.ErrUns_S2{Iter} = ErrUns_S2;
    sd.ErrSta_S1{Iter} = ErrSta_S1;
    sd.ErrUns_S1{Iter} = ErrUns_S1;
    sd.ErrSta_Sinf{Iter} = ErrSta_Sinf;
    sd.ErrUns_Sinf{Iter} = ErrUns_Sinf;
            
    %% Print intermediate results
    fprintf('k+l = %d, NyRelErr = %f, NyRelErr = %f  ... \n', ... 
         T(Iter), ErrSta_S2./ErrBest_S2-1, ErrUns_S2./ErrBest_S2-1 );
    
end

%% Variables to be saved
sd.ErrBest_S2 = ErrBest_S2;
sd.ErrBest_S1 = ErrBest_S1;
sd.ErrBest_Sinf = ErrBest_Sinf;
sd.info.n = n;
sd.info.r = r;
sd.info.field = field;
sd.info.T = T;
sd.ID = [datestr(now,30),num2str(randi([100,999]))];

end