%Our primary goal is to obtain the best possible reconstruction of an input
%matrix using the smallest sketch possible. For standard normal test
%matrices, the storage cost of the sketch is directly proportional to the
%sum T = (k + l) of the sketch size parameters. In this experiments, we
%investigate the best way to apportion k and l when we fix the sketch size
%T and the target rank r.
%
%For different values of (T = k+l), we sweep over all possible (k,l) pairs,  
%and record the approximation error, norm(A - Aapprox,'fro').
%
%Test_generic_Type1 and Test_generic_Type2 files are different in the way
%input matrix A is applied. Type1 uses a dense matrix A. Type 2 computes
%the sketch of a low-rank matrix A=Hf*Hb', directly from its factors Hf
%and Hb.
%
%In addition to the algorithms we recommend, our numerical study comprises 
%other methods that have appeared in the literature. See Section 7.3 for  
%the details about these algorithms. 
%
%   See our reference paper for the detailed explanation of the
%   sketching procedure and the arithmetic, communication and storage
%   costs.
%
%   [TYUC2017] J.A. Tropp, A. Yurtsever, M. Udell and V. Cevher, Practical  
%   Sketching Algorithms for Low-Rank Matrix Approximation, 2017.
%
%   Coded by: Alp Yurtsever
%   Ecole Polytechnique Federale de Lausanne, Switzerland.
%   Laboratory for Information and Inference Systems, LIONS.
%   contact: alp.yurtsever@epfl.ch
%   Created: April 12, 2017
%   Last modified: October 24, 2017
%
%Nys-SKETCHv1.0
%Copyright (C) 2017 Laboratory for Information and Inference Systems
%(LIONS), Ecole Polytechnique Federale de Lausanne, Switzerland.
%This code is a part of Nys-SKETCH toolbox. 
%Please read COPYRIGHT before using this file.

%% Beginning of the test

T = unique([r:2:2*r, round((2*r+3) * 2.^(0:.5:4))]);
if r == 1
    T = unique([r:2:2*r, round((2*r+3) * 2.^(0:.5:5))]);
end


for Iter = 1:numel(T)
    
    %% NYSTROM Sketch
    myNystromSketch = NystromSketch(n,T(Iter),field,model);
    myNystromSketch = myNystromSketch.LinearUpdate(0, 1, U*Z, U);
    [QN, DN] = myNystromSketch.FixedRankPSDApprox(r);
    [~,ZZ] = updateThinSVD(U,Z,U,QN*(-DN),QN);
    ErrNy_S2  = norm(diag(ZZ));
    ErrNy_S1  = norm(diag(ZZ),1);
    ErrNy_Sinf  = norm(diag(ZZ),inf);
            
    %% GITTNES&MAHONEY reconstruction
    [QN, DN] = myNystromSketch.GittensMahoneyApprox(r);
    [~,ZZ] = updateThinSVD(U,Z,U,QN*(-DN),QN);
    ErrGM_S2  = norm(diag(ZZ));
    ErrGM_S1  = norm(diag(ZZ),1);
    ErrGM_Sinf  = norm(diag(ZZ),inf);

    %% Preallocation
    sd.k{Iter} = [];
    sd.l{Iter} = [];
    sd.ErrA9_S2{Iter} = [];
    sd.ErrA9_S1{Iter} = [];
    sd.ErrA9_Sinf{Iter} = [];
    sd.ErrNy_S2{Iter} = ErrNy_S2;
    sd.ErrNy_S1{Iter} = ErrNy_S1;
    sd.ErrNy_Sinf{Iter} = ErrNy_Sinf;
    sd.ErrGM_S2{Iter} = ErrGM_S2;
    sd.ErrGM_S1{Iter} = ErrGM_S1;
    sd.ErrGM_Sinf{Iter} = ErrGM_Sinf;

    %% Sweep parameters
    k = r;
    l = T(Iter) - k;

    while r <= l
        %% Our Single-View Fixed-Rank Approximation with structures
        mySketch = Sketch([n, n], k, l, field, model);
        mySketch = mySketch.LinearUpdate(U*Z, U, 0, 1);
        
        if (k < l) && (r < k)            
            %% Our Single-View Fixed rank PSD Approximation (Alg.9)
            [Q9, Z9] = mySketch.FixedRankPSDApprox(r);
            [~,ZZ] = updateThinSVD(U,Z,U,Q9*(-Z9),Q9);
            ErrA9_S2  = norm(diag(ZZ));
            ErrA9_S1  = norm(diag(ZZ),1);
            ErrA9_Sinf  = norm(diag(ZZ),inf);
        else
            ErrA9_S2 = nan;
            ErrA9_S1 = nan;
            ErrA9_Sinf = nan;
        end
        
        %% Save data
        sd.k{Iter}(end+1) = k;
        sd.l{Iter}(end+1) = l;
        sd.ErrA9_S2{Iter}(end+1) = ErrA9_S2;
        sd.ErrA9_S1{Iter}(end+1) = ErrA9_S1;
        sd.ErrA9_Sinf{Iter}(end+1) = ErrA9_Sinf;

        %% increase k, decrease l, keep k+l constant
        k = k+1;
        l = l-1;
    end
    
    %% Print intermediate results
    fprintf('k+l = %d, NyRelErr = %f, GMRelErr = %f, A9RelErr = %f ... \n', ... 
         T(Iter), ErrNy_S2./ErrBest_S2-1, ErrGM_S2./ErrBest_S2-1, min(sd.ErrA9_S2{Iter})./ErrBest_S2-1 );

end

%% Variables to be saved
sd.ErrBest_S2 = ErrBest_S2;
sd.ErrBest_S1 = ErrBest_S1;
sd.ErrBest_Sinf = ErrBest_Sinf;
sd.info.n = n;
sd.info.n = n;
sd.info.r = r;
sd.info.field = field;
sd.info.T = T;
sd.ID = [datestr(now,30),num2str(randi([100,999]))];

