classdef Sketch
    %SKETCH of a (m x n) dimensional target matrix A takes the form
    %(Y = A*Omega) and (W = Psi*A) which are (m x k) and (l x n)
    %dimensional respectively. Here (n x k) dimensional matrix (Omega) and
    %(l x m) dimensional matrix (Psi) are called as the Test Matrices. This
    %codes implements the sketching method proposed in the reference paper,
    %where the test matrices are statistically independent and follow the
    %standard normal distribution. The matrix (Y) collects information
    %about the range of (A), while the matrix (W) collects information
    %about the co-range of (A). Both parts are necessary.
    %
	%   NOTICE: THIS FILE IS MODIFIED FROM THE PRACTICALSKETCHING TOOLBOX 
	%   by Alp Yurtsever to add SSFT test matrices.
    %
    %See our reference paper [TYUC2017P] for the detailed explanation of the
    %sketching procedure and the arithmetic, communication and storage
    %costs.
    %
	%[TYUC2017P] J.A. Tropp, A. Yurtsever, M. Udell and V. Cevher, Practical  
	%Sketching Algorithms for Low-Rank Matrix Approximation, SIMAX, 2017.
    %
    %Coded by: Alp Yurtsever
    %Ecole Polytechnique Federale de Lausanne, Switzerland.
    %Laboratory for Information and Inference Systems, LIONS.
    %contact: alp.yurtsever@epfl.ch
    %Created: June 28, 2016
    %Last modified: October 24, 2017
    %
    %PRACTICALSKETCHING-v1.0
    %Copyright (C) 2017 Laboratory for Information and Inference Systems
    %(LIONS), Ecole Polytechnique Federale de Lausanne, Switzerland.
    %
    %This code is a part of PRACTICALSKETCHING toolbox.
    %Please read COPYRIGHT before using this file.
    
    %% properties
    properties
        Field     % 'real' or 'complex' field (default, 'complex')
        Model     % 'gaussian', 'orthonormal', or 'ssft' etc
        Omega     % (n x k) dimensional test matrix for the range of A
        Psi       % (l x m) dimensional test matrix for the corange of A
        Y         % (m x k) dimensional range sketch
        W         % (l x n) dimensional corange sketch
        m, n      % dimensions of data
        k, l      % dimensions of sketch
    end
    
    %% methods
    methods
        % Constructor
        % function obj = Sketch(A, k, l, Field, Orthogonalization)
        function obj = Sketch(A, k, l, varargin)
            % process variable length inputs
            obj.k = k; obj.l = l;
            % 1. Field, 2. Orthogonalization
            narginchk(3,5);
            if nargin == 4
                obj.Field  = lower(varargin{1});
            else
                obj.Field = 'complex';
            end
            if nargin == 5
                obj.Model  = lower(varargin{2});
            else
                obj.Model = 'gaussian';
            end
            % construct the sketch
            [m, n] = size(A);
            if (m + n) == 3
                m = A(1);
                n = A(2);
                A = [];
            end
            obj.m = m; obj.n = n;
            if strcmp(obj.Model, 'ssft')
                indR1 = sort(randsample(n,k));
                opR1  = @(x) x(:,indR1);
                Pi11  = spdiags(sign((randn(n,1)<0)-0.5),0,n,n);
                Pi11  = Pi11(randperm(n),:);
                Pi12  = spdiags(sign((randn(n,1)<0)-0.5),0,n,n);
                Pi12  = Pi12(randperm(n),:);
                indR2 = sort(randsample(m,l));
                opR2  = @(x) x(indR2,:);
                Pi21  = spdiags(sign((randn(n,1)<0)-0.5),0,n,n);
                Pi21  = Pi21(randperm(n),:);
                Pi22  = spdiags(sign((randn(n,1)<0)-0.5),0,n,n);
                Pi22  = Pi22(randperm(n),:);
                if strcmp(obj.Field,'real')
                    dctLeftMult = @(x) idct(x')';
                    obj.Omega = @(x) opR1(dctLeftMult(dctLeftMult(x*Pi11)*Pi12));
                    obj.Psi = @(x) opR2(dct(Pi22*dct(Pi21*x)));
                elseif strcmp(obj.Field,'complex')
                    dftLeftMult = @(x) (ifft(full(x'))*sqrt(n))';
                    uni_fft = @(x) fft(full(x))/sqrt(n);
                    obj.Omega = @(x) opR1(dftLeftMult(dftLeftMult(x*Pi11)*Pi12));
                    obj.Psi = @(x) opR2(uni_fft(Pi22*uni_fft(Pi21*x)));
                else
                    error('Field should be ''real'' or ''complex''.');
                end               
            elseif strcmp(obj.Model, 'orthonormal') || strcmp(obj.Model, 'gaussian')
                if strcmp(obj.Field,'real')
                    obj.Omega = randn(n,k);
                    obj.Psi   = randn(l,m);
                elseif strcmp(obj.Field,'complex')
                    obj.Omega = randn(n,k) + 1i*randn(n,k);
                    obj.Psi   = randn(l,m) + 1i*randn(l,m);
                else
                    error('Field should be ''real'' or ''complex''.');
                end
                if strcmp(obj.Model, 'orthonormal')
                    obj.Omega = orth(obj.Omega);
                    obj.Psi   = orth(obj.Psi')';
                end
            else
                error('Model should be ''gaussian'', ''orthonormal'', or ''ssft''.');
            end
            % initialize the sketch
            if isempty(A)
                obj.Y = zeros(m,k);
                obj.W = zeros(l,n);
            else
                if isa(obj.Omega,'function_handle')
                    obj.Y = obj.Omega(A);   % Note that Omega(A) computes A*Omega
                    obj.W = obj.Psi(A);
                else
                    obj.Y = A*obj.Omega;
                    obj.W = obj.Psi*A;
                end
            end
            
        end
        
        % Other methods
        obj = LinearUpdate(obj, varargin)           % Algorithm 2 in [MR]
        [Q, X] = LowRankApprox(obj)                 % Algorithm 4
        [U, S] = LowRankSymApprox(obj)              % Algorithm 5
        [U, D] = FixedRankPSDApprox(obj, r)         % Algorithm 9
    end
end


