function obj = LinearUpdate( obj, varargin )
%LINEARUPDATE Single-View Sketch: Linear Update
%This function implements (3.6) from the main reference.
%   Require: (m x n) dimensinal update matrix (H)
%            scalars (theta) and (eta)
%   Ensure:  Modifies sketch (Y,W) to reflect linear update
%                    A = theta*A + eta*H
%            which takes the form
%                    Y = theta*Y + eta*H*Omega
%                    W = theta*W + eta*Psi*H
%
%   S = S.LinearUpdate(H, theta, eta) updates the sketches Y and W as 
%   Y = theta*Y + eta*H*Omega and W = theta*W + eta*Psi*H.
%
%   S = S.LinearUpdate(U, V, theta, eta) where Hf and Hb are tall
%   matrices updates the sketches Y and W from the factors H = U*V' 
%   in a storage efficient way.
%   
%   S = S.LinearUpdate(Hforw, Hback, theta, eta) where Hforw and Hback 
%   are the function handles to compute H*x = Hforw(x) and x*H = Hback(x),
%   updates the sketches Y and W as Y = theta*Y + eta*H*Omega and 
%   W = theta*W + eta*Psi*H.
%
%See our reference paper for the detailed explanation of the sketching
%procedure and the arithmetic, communication and storage costs.
%   
%[TYUC2017P] J.A. Tropp, A. Yurtsever, M. Udell and V. Cevher, Practical  
%Sketching Algorithms for Low-Rank Matrix Approximation, 2017.
%
%Coded by: Alp Yurtsever
%Ecole Polytechnique Federale de Lausanne, Switzerland.
%Laboratory for Information and Inference Systems, LIONS.
%contact: alp.yurtsever@epfl.ch
%Created: June 28, 2016
%Last modified: May 12, 2017

narginchk(4,5);
if nargin == 4
    H       = varargin{1};
    theta   = varargin{2};
    eta     = varargin{3};
    if isa(obj.Omega,'function_handle')
        obj.Y   = theta*obj.Y + eta*(obj.Omega(H)); % Note that Omega(H) computes H*Omega
        obj.W   = theta*obj.W + eta*(obj.Psi(H)); % Note that Psi(H) computes Psi*H
    else
        obj.Y = theta*obj.Y + eta*(H*obj.Omega);
        obj.W = theta*obj.W + eta*(obj.Psi*H);
    end
else
    Hforw   = varargin{1};
    Hback   = varargin{2};
    theta   = varargin{3};
    eta     = varargin{4};
    if isa(Hforw,'function_handle')
        if isa(obj.Omega,'function_handle')
            warning(['Both ''H'' and test matrix ''Omega'' are function handles. ' ...
                'This case is not supported, so we compute Omega in full dimensions.'])
            obj.Y   = theta*obj.Y + eta*(Hforw(obj.Omega(eye(obj.n))));
            obj.W   = theta*obj.W + eta*(Hback(obj.Psi(eye(obj.m))));
        else
            obj.Y   = theta*obj.Y + eta*(Hforw(obj.Omega));
            obj.W   = theta*obj.W + eta*(Hback(obj.Psi));
        end
    else
        if isa(obj.Omega,'function_handle')
            obj.Y = theta*obj.Y + eta*(Hforw*(obj.Omega(Hback')));  % Note that Omega(Hback') computes Hback'*Omega
            obj.W = theta*obj.W + eta*((obj.Psi(Hforw))*Hback');    % Note that Psi(Hforw) computes Psi*Hforw
        else
            obj.Y = theta*obj.Y + eta*(Hforw*(Hback'*obj.Omega));
            obj.W = theta*obj.W + eta*((obj.Psi*Hforw)*Hback');
        end
    end
end

end

