function obj = LinearUpdate( obj, varargin )
%LINEARUPDATE Nystrom Sketch for Fixed-Rank Approximation of a PSD Matrix
%This function implements a linear update routine for the sketches 
%described in [TYUC2017Nys].
%   Require: (n x n) dimensinal update matrix (H)
%            scalars (theta1) and (theta2)
%   Ensure:  Modifies sketch (Y) to reflect linear update
%                    A = theta1*A + theta2*H
%            which takes the form
%                    Y = theta1*Y + theta2*H*Omega
%
%   S = S.LinearUpdate(theta1, theta2, H) updates the sketch Y. H can be a
%   matrix of the function handle of a linear operator.
%
%   S = S.LinearUpdate(theta1, theta2, U, V) where U and V are tall
%   matrices updates the sketches Y from the factors H = U*V' in a storage
%   efficient way.
%
%[TYUC2017Nys] J.A. Tropp, A. Yurtsever, M. Udell and V. Cevher. Fixed-Rank 
%Approximation of a Positive-Semidefinite Matrix from Streaming Data.
%In Proc. 31st Conference on Neural Information Processing Systems
%(NIPS), Long Beach, CA, USA, December 2017.
%
%Coded by: Alp Yurtsever
%Ecole Polytechnique Federale de Lausanne, Switzerland.
%Laboratory for Information and Inference Systems, LIONS.
%contact: alp.yurtsever@epfl.ch
%Created: April 12, 2017
%Last modified: October 21, 2017
%
%Nys—SKETCHv1.0
%Copyright (C) 2017 Laboratory for Information and Inference Systems
%(LIONS), Ecole Polytechnique Federale de Lausanne, Switzerland.
%This code is a part of Nys—SKETCH toolbox. 
%Please read COPYRIGHT before using this file.

narginchk(4,5);
theta1   = varargin{1};
theta2   = varargin{2};
if nargin == 4
    H       = varargin{3};
    if isa(H,'function_handle')
        if isa(obj.Omega,'function_handle')
            warning(['Both ''H'' and test matrix ''Omega'' are function handles. ' ...
                'This case is not supported, so we compute Omega in full dimensions.'])
            obj.Y   = theta1*obj.Y + theta2*(H(obj.Omega(eye(size(H,2))))); % Note that Omega(H) computes H*Omega
        else
            obj.Y   = theta1*obj.Y + theta2*(H(obj.Omega));
        end
    else
        if isa(obj.Omega,'function_handle')
            obj.Y   = theta1*obj.Y + theta2*(obj.Omega(H)); % Note that Omega(H) computes H*Omega
        else
            obj.Y   = theta1*obj.Y + theta2*(H*obj.Omega);
        end
    end
else
    U       = varargin{3};
    V       = varargin{4};
    if isa(obj.Omega,'function_handle')
        obj.Y   = theta1*obj.Y + theta2*(U*(obj.Omega(V'))); % Note that Omega(V') computes V'*Omega
    else
        obj.Y   = theta1*obj.Y + theta2*(U*(V'*obj.Omega));
    end
end

end
