function [xsol, output] = inexact_admm_sqrt_lasso(A, b, lambda, param)

    % Check the inputs.
    if nargin < 3, error('At least inputs must be provided!'); end
    if nargin < 4, param = []; end
    param = lassoParamSettings(param);
    
    % Initialize the variables.
    output.time = tic;
    nx          = size(A, 1);
    px          = size(A, 2);
    norm_b      = norm(b, 2);

    % Define two prox-operators.
    prox2Norm = @(x, lambda)( max(1.0 - lambda/norm(x, 2), 0).*x );
    sofThhOpr = @(x, lambda)( sign(x).*max(abs(x) - lambda, 0) );
    
    % Define the linear operator.
    Aoper     = @(x)(A*x);
    AToper    = @(x)(A'*x);

    % Define the starting point.
    y_cur     = zeros(nx, 1);
    x_cur     = zeros(px, 1);
    r_cur     = y_cur;
    rel_feas  = 0;
    rel_schg  = 0;
    abs_dfeas = 0;
    Fx_val    = inf;
    
    % Initialize the parameters.
    %Lips     = max(real(eig(A'*A)));
    Lips     = lassoNormAtAeval('operator', Aoper, 20, 1e-5, AToper, px);
    iLips    = 1.0/Lips;
    rho      = min(max(0.1/Lips, 1.0e-6), 1.0); 
    min_rho  = 1e-8;
    max_rho  = 1.0e8;
    rhofact  = 1.1;
    incrrho  = 1.05;
    decrrho  = 1.00;

    % Applying the linear operator.
    Ax_cur   = Aoper(x_cur);
    cntA     = 1;
    cntAt    = 0;
    
    % The main loop.
    for iter = 1:param.MaxIters

        % Evaluate the objective value.
        Fx_val   = norm(Ax_cur - b, 2) + lambda*norm(x_cur, 1);
        
        % Compute the r-component.
        irho     = 1.0/rho;
        r_center = Ax_cur - b + irho*y_cur;
        r_next   = prox2Norm(r_center, irho);

        % Compute the x-component.
        x_center = Ax_cur - r_next + irho*y_cur - b;
        iLATrc   = iLips*AToper(x_center);
        cntAt    = cntAt + 1;
        x_next   = sofThhOpr(x_cur - iLATrc, lambda*irho*iLips);

        % Applying the linear operator.
        Ax_next  = Aoper(x_next);
        cntA     = cntA + 1;
        
        % The feasibility.
        feas     = Ax_next - r_next - b;

        % Update the dual variable y.
        y_next   = y_cur + rho*feas;

        % Compute the change nor solution norm and feasibility gap.
        abs_schg = norm(x_next - x_cur);
        rel_schg = abs_schg/max(1.0, norm(x_cur, 2));
        abs_feas = norm(feas, 2);
        rel_feas = abs_feas/max(1.0, norm_b);

        % Print the iteration.
        if param.Verbosity > 1, printIteration(param.PrintStep); end
        
        % Save the history if required.
        if param.saveHistMode > 0, saveHistoryList(); end
        
        % Check the stopping criterion.
        if rel_feas <= param.RelTolX && rel_schg <= param.RelTolX*1e3
            output.status = 'Convergence achieved!';
            break;
        end

        % Update the penalty parameter.
        updatePenPar();
        
        % Go to the next iterations.
        x_cur  = x_next;
        y_cur  = y_next;
        r_cur  = r_next;
        Ax_cur = Ax_next;
    end

    % Get the final solution.
    xsol           = x_next;
    output.iter    = iter;
    output.Fx_val  = Fx_val;
    output.relGap  = rel_feas;
    output.relErr  = rel_schg;
    output.absDgap = abs_dfeas;
    output.time    = toc(output.time);
    output.cntA    = cntA;
    output.cntAt   = cntAt;
    
    
%%% IMPLEMENTATION OF THE NEST FUNCTIONS.
%%% -----------------------------------------------------------------------
%%% Function: updatePenPar
%%% Purpose:  Update the penalty parameter.
%%% -----------------------------------------------------------------------
    function updatePenPar()
        
            abs_dfeas = rho*norm(AToper(r_next - r_cur), 2);
            cntAt     = cntAt + 1;
            
            if abs_feas >= rhofact*abs_dfeas
                rho = incrrho*rho;
            elseif abs_dfeas >= rhofact*abs_feas
                rho = rho/decrrho;
            end
            
            rho = min(max(rho, min_rho), max_rho);
    end

%%% -----------------------------------------------------------------------
%%% Function: printIteration(step)
%%% Purpose: Print the output at each iteration.
%%% -----------------------------------------------------------------------
    function printIteration(step)
        if mod(iter, 10*step) == 1
            fprintf('%s\n', repmat('-', 1, 50));
            fprintf('Iter|  FeasGap |  Schange |  PenPar  |  F(x) \n');
            fprintf('%s\n', repmat('-', 1, 50));
        end
        if mod(iter, step) == 0
            fprintf('%4d| %3.3e| %3.3e| %3.3e| %3.5e\n', ...
                    iter, rel_feas, rel_schg, rho, Fx_val);
        end
    end
    
%%% -----------------------------------------------------------------------
%%% Function: saveHistoryList
%%% Purpose:  Save the history information if required.
%%% -----------------------------------------------------------------------
    function saveHistoryList()
       output.hist.rel_schg(iter,1)  = rel_schg;
       output.hist.rf_gap(iter,1)    = rel_feas;
       output.hist.fx_val(iter,1)    = Fx_val;
       if param.saveHistMode > 1
           output.hist.rho(iter,1)   = rho;
       end
       if param.saveHistMode > 2
           output.hist.xb{iter}  = x_cur;
           output.hist.yb{iter}  = y_cur;
       end
       
    end
end