function y = l1ball_projection(x, tau)
% x = l1ball_projection(y, tau) - Compute x such that
% 
%   min_{x} .5*||y - x||_2^2   s.t.  f(x) <= tau,
% 
% where f(x) = sum_{i, j} |x(i, j)| with 
% |x(i, j)| = sqrt(abs(x(:, :, 1)).^2 + abs(x(:, :, 2)).^2).
% 
% 
% The implementation of this function is based on the following papers:
% [1] J.M. Fadili, G. Peyre, "Total variation projection with first order 
% schemes" , IEEE Transactions on Image Processing, vol. 20(3), 
% pp. 657-669, 2011.
% [2] J.M. Fadili, G. Peyre, "Total variation projection with first order 
% schemes", Proc. of the 16th IEEE Int. Conf. on Image Processing, 
% pp. 1325-1328, 2009.
% 
% The original implementation of this function is available on the personal 
% webpage of J.M. Fadili and G. Peyre:
% http://www.ceremade.dauphine.fr/%7Epeyre/download/tv-projection.zip
% 
% This implementation is given for compatibility purpose with the current
% CLASH implementation.
% 
% 
% Author: Gilles Puy
% E-mail: gilles.puy@epfl.ch
% Date: Sep. 13, 2011

if tau == 0, y = x*0; return; end

% Ordered norms and cumulative ordered norms
abs_x = sqrt(abs(x(:, :, 1)).^2 + abs(x(:, :, 2)).^2);
d = sort(abs_x(:), 'ascend'); % Ordered norm
D = flipud(cumsum(flipud(d))); % Cumulated ordered norm
D = D - d.*(numel(abs_x):-1:1)'; %

% Find  "t"  s.t D(t) > tau >= D(t+1)
ind = find(D > tau, 1, 'last');

% Threshold 
lambda = d(ind+1) + (d(ind) - d(ind+1)) * (D(ind+1) - tau)/(D(ind+1)-D(ind));

% Soft-thresholding
y = repmat(max(1 - lambda./abs_x, 0), [1 1 2]).*x;

end


