function xsol = TVoPsi_projection(y, lambda, Psi, Psit, param)
% x = TVoPsi_projection(y, lambda, Psi, Psit, param) - Find x such that:
% 
%       min_{x} .5*||y - x||_2^2   s.t.  ||Psi*x||_{TV} <= lambda 
% 
% 
% The implementation of this function is based on the following papers:
% [1] J.M. Fadili, G. Peyre, "Total variation projection with first order 
% schemes" , IEEE Transactions on Image Processing, vol. 20(3), 
% pp. 657-669, 2011.
% [2] J.M. Fadili, G. Peyre, "Total variation projection with first order 
% schemes", Proc. of the 16th IEEE Int. Conf. on Image Processing, 
% pp. 1325-1328, 2009.
% [3] A. Beck, M. Teboulle, "A Fast Iterative Shrinkage-Thresholding
% Algorithm for Linear Inverse Problems", SIAM Journal on Imaging Sciences
% vol. 2(1), pp. 183-202, 2009.
% 
% The original original implementation where Psi is the identity is 
% available on the personal webpage of J.M. Fadili and G. Peyre:
% http://www.ceremade.dauphine.fr/%7Epeyre/download/tv-projection.zip
% 
% 
% Author: Gilles Puy
% E-mail: gilles.puy@epfl.ch
% Date: Sept. 12, 2011

% Optional input arguments
if ~isfield(param, 'verbose'), param.verbose = 1; end
if ~isfield(param, 'rel_obj'), param.rel_obj = 1e-5; end
if ~isfield(param, 'max_iter'), param.max_iter = 500; end
if ~isfield(param, 'Lip'), param.Lip = 8; end % Lipschitz constant

% Initializations
w = repmat(Psi(y).*0, [1, 1, 2]); xold = w; xsol = y.*0; told = 1; iter = 1;

%
while 1
    
    % Current solution
    token = xsol; xsol = y - Psit_div(Psit, w);
    
    % Stopping criterion
    rel_obj = norm(token(:) - xsol(:))/norm(xsol(:));
    if param.verbose >= 1
        fprintf([' Iter %i:, rel_obj = %1.5e, ' ...
            'lambda = %1.5e, TV = %e\n'], iter, rel_obj, lambda, ...
            TV_norm(Psi(xsol), 0));
    end
    if iter >= param.max_iter || abs(rel_obj) < param.rel_obj
        break;
    end    
    
    % Gradient step
    dummy = w - 1/param.Lip * grad_Psi(Psi, xsol);
    
    % L_inf prox
    xhat = dummy - l1ball_projection(dummy, lambda/param.Lip);
    
    % Update
    t = (1+sqrt(1+4*told^2))/2;
    w = xhat + (told-1)/t * (xhat-xold);
    told = t; xold = xhat; iter = iter + 1;
    
end

end