function xhat = CLASH_TV(yy, A, At, Psi, Psit, lambda, K, Its, tol, mode, im, L)

% Concatenate operators
Phi = @(x) A(Psi(x));
Phit = @(x) Psit(At(x));

% Initializations
kk = 1; % Current iteration
ws_vec = Phit(yy).*0; % Initial wavelet vector
[M, N] = size(ws_vec); % Signal size
supp= []; % Current support
active_X = ones(size(ws_vec)); % Active set

% Parameter for constrained TV
param.Lip = 30; % Lipschitz constant
param.verbose = 1;
param.rel_obj = 1e-4;
param.max_iter = 500;
param.max_iter_TV = 500;


while kk<=Its
    
    % Keep in memory current estimate
    ttoken = ws_vec;
    
    % Approximate gradient in model space
    rr = yy - Phi(ws_vec); proxy = Phit(rr); % Gradient
    active_X(supp) = 0;
    proxy_im = proxy.*active_X; % Gradient on support complement
    active_X(supp)= 1;
    
    % Find best 2K sparse approximation
    if strcmp(mode, 'regular')
        [~, ind] = sort(abs(proxy_im(:)),'descend');
        tt = union(supp, ind(1:K));
    elseif strcmp(mode, 'tree')
        Bn = abs(proxy_im).^2;
        maskn = cssa2(Bn, L, K);
        tt = union(supp, find(maskn(:)>0));
    end
    
    % Restricted objective minimization
    %   Define restricted sparsity operators
    PsiR = @(x) opRes(x, Psi, tt, [M, N]);
    PsiRt = @(x) optRes(x, Psit, tt);
    %   Define restricted (measurement o sparsity) operator
    PhiR = @(x) opRes(x, Phi, tt, [M, N]);
    PhiRt = @(x) optRes(x, Phit, tt);
    %   Solve constrained TV
    xx = solve_restricted_TV(yy, lambda, PhiR, PhiRt, PsiR, PsiRt, param);
    
    % Find best K sparse approximation
    bb = ws_vec.*0; bb(tt)= xx;
    if strcmp(mode, 'regular')
        [~, ind] = sort(abs(bb(:)),'descend');
        tt = ind(1:K);
    elseif strcmp(mode, 'tree')
        Bn = abs(proxy_im).^2;
        maskn = cssa2(Bn, L, K);
        tt = find(maskn(:)>0);
    end
    
    % Restricted objective minimization
    %   Define restricted sparsity operators
    PsiR = @(x) opRes(x, Psi, tt, [M, N]);
    PsiRt = @(x) optRes(x, Psit, tt);
    %   Define restricted (measurement o sparsity) operator
    PhiR = @(x) opRes(x, Phi, tt, [M, N]);
    PhiRt = @(x) optRes(x, Phit, tt);
    %   Solve constrained TV
    xx = solve_restricted_TV(yy, lambda, PhiR, PhiRt, PsiR, PsiRt, param);
    
    % Current estimate
    ws_vec = ws_vec.*0; ws_vec(tt) = xx; 
    supp = tt;
    
    % Stopping criterion
    imagesc(real(Psi(ws_vec))); axis image; axis off; colormap gray; drawnow
    disp(SNR(im, real(Psi(ws_vec))));
    if norm(ws_vec - ttoken) < tol*norm(ws_vec)
        disp('yahoo');
        break;
    end
    kk = kk+1;
    
end

% Final solution
xhat = Psi(ws_vec);

end

function xhat = solve_restricted_TV(y, lambda, A, At, Psi, Psit, param)
% x = solve_restricted_TV(y, lambda, A, At, Psi, Psit, param) - Find x 
% such that:
% 
%       min_{x} .5*||y - A*x||_2^2   s.t.  ||Psi*x||_{TV} <= lambda 
% 
% 
% The implementation of this function is based on the following papers:
% [1] J.M. Fadili, G. Peyre, "Total variation projection with first order 
% schemes" , IEEE Transactions on Image Processing, vol. 20(3), 
% pp. 657-669, 2011.
% [2] J.M. Fadili, G. Peyre, "Total variation projection with first order 
% schemes", Proc. of the 16th IEEE Int. Conf. on Image Processing, 
% pp. 1325-1328, 2009.
% [3] A. Beck, M. Teboulle, "A Fast Iterative Shrinkage-Thresholding
% Algorithm for Linear Inverse Problems", SIAM Journal on Imaging Sciences
% vol. 2(1), pp. 183-202, 2009.
% 
% Original implementation of the TV projection (with Psi identity) is 
% available on the personal webpage of J.M. Fadili and G. Peyre:
% http://www.ceremade.dauphine.fr/%7Epeyre/download/tv-projection.zip
% 
% 
% Author: Gilles Puy
% E-mail: gilles.puy@epfl.ch
% Date: Sept. 12, 2011


% Optional input arguments
if ~isfield(param, 'Lip'), param.Lip = 1; end
if ~isfield(param, 'verbose'), param.verbose = 1; end
if ~isfield(param, 'rel_obj'), param.rel_obj = 1e-4; end
if ~isfield(param, 'max_iter'), param.max_iter = 500; end
if isfield(param, 'max_iter_TV')
    param_TV.max_iter = param.max_iter_TV;
end
param_TV.rel_obj = param.rel_obj;
param_TV.verbose = param.verbose - 1;


% Init
w = zeros(size(At(y))); xold = zeros(size(At(y)));
told = 1; t = 1; previous_obj = 1/2*norm(y)^2; iter = 1;

while 1
    
    % Gradient step
    dummy = w + 1/param.Lip * At(y-A(w));
    
    % TV projection
    if TV_norm(Psi(dummy))>lambda
        xhat = TVoPsi_projection(dummy, lambda, Psi, ...
            Psit, param_TV);
    else
        xhat = dummy;
    end
    
    % Stopping criterion
    obj = .5*norm(y-A(xhat))^2;
    rel_obj = abs(previous_obj-obj)/previous_obj;
    if param.verbose >= 1
        fprintf([' Iter %i:, obj = %1.5e, rel_obj = %1.5e, ' ...
            'lambda = %1.5e, TV = %e\n'], iter, obj, rel_obj, lambda, ...
            TV_norm(Psi(xhat)));
    end
    if iter>=param.max_iter || abs(rel_obj) < param.rel_obj
        break;
    end
    
    % Update
    t = (1+sqrt(1+4*told^2))/2;
    w = xhat + (told-1)/t * (xhat-xold);
    told = t; xold = xhat; previous_obj = obj; iter = iter + 1;
    
end

end
