%%  Kernel Density Estimate
% This function calculates a probability density estimate from the current
% blockages of the signal. It then tries to estimate the best Gaussian fit by
% evaluating the likelihood of a Gaussian mixture model with x Gaussians.
% Please evaluate the result by looking at the Log-Likelihood and BIC/AIC
% values for different numbers of Gaussians. Input variables are
% EventDatabase and maxGaussians, which denotes the maximal number of
% Gaussians to fit.
%
% Use:
% densityest(maxGaussians). The script prompts you to choose an event
% database automatically.

function densityest(maxGaussians)

%% IMPORT
[FileName,PathName,FilterIndex] = uigetfile('*.mat','Select event database...');
a=load([PathName FileName]);
if(~isfield(a, 'EventDatabase'))
    errordlg('Please specify a .mat file containing an event database','Error')
    return
end
EventDatabase=a.EventDatabase;
current_units=a.current_units;
%%

close all
L=EventDB2LTable(EventDatabase);
figure
[f xi]=ksdensity(L(:,6));
plot(xi,f);
xlabel(['Current Drop [' current_units ']']);
ylabel('Density');
title('Density plot');

options = statset('MaxIter',300);
liklMat=[];

for i=1:maxGaussians
obj = gmdistribution.fit(L(:,6),i,'Options',options);
liklMat=[liklMat; obj.NlogL,obj.AIC,obj.BIC];
end

%Plot log-likelihoods
figure
plot(1:maxGaussians,liklMat(:,1));
xlabel('Number of components');
ylabel('Log-Likelihood');
title('Log-Likelihood vs. Number of Components')

figure
plot(1:maxGaussians,liklMat(:,2:3));
xlabel('Number of components');
ylabel('BIC/AIC');
legend('AIC','BIC')
title('AIC/BIC Values vs. Number of Components')


BIC_values=liklMat(:,3);
temp=Inf;
for j=1:length(BIC_values)
    if (BIC_values(j)<temp)
        best=j;
        temp=BIC_values(j);

    end
end

figure
obj = gmdistribution.fit(L(:,6),best,'Options',options);
h=pdf(obj,xi');
plot(xi,f);
hold on;
plot(xi,h,'r');
xlabel('Current Drop');
ylabel('Density');
legend('Kernel Density Estimate','Gaussian Mixture Distribution')
title('Density Distribution + Gaussian Fit')

fprintf('The data is best fit with %i Gaussians\n',best);
end
