%%  Function RemoveEventsFromRaw:
%   Use this function to remove individual events from the raw signal. The first input
%   is the Raw Signal of the analysis, the second input is an vector or
%   scalar of event numbers to be removed from the database. The third
%   argument is optional and provides an alternative way of removing
%   events. Instead of removing events based on their event number, the
%   'Threshold' method removes all events longer than a threshold given in
%   argument two. This threshold is given in the time domain based on units and
%   sampling frequencies of the analysis.
%   Use:
%   1) newRaw=RemoveEventsFromRaw(RawSignal, numbers)
%   2) newRaw=RemoveEventsFromRaw(RawSignal, threshold,'Threshold')


function newRawSignal=RemoveEventsFromRaw(RawSignal,numbers,varargin)
[FileName,PathName,FilterIndex] = uigetfile('*.mat','Select event database...');
a=load([PathName FileName]);
if(~isfield(a, 'EventDatabase'))
errordlg('Please specify a .mat file containing an event database','Error')
return
end
EventDatabase=a.EventDatabase;
ConcatenatedEvents=a.ConcatenatedEvents;
ConcatenatedFits=a.ConcatenatedFits;
IncludedBaselinePlots=a.IncludedBaselinePlots;
delta=a.delta;
sigma=a.sigma;
SamplingFrequency=a.SamplingFrequency;
time_units=a.time_units;
units=1e3;
switch time_units
    case 'us' 
        units=1e6;
    case 'ms' 
        units=1e3;
    case 's' 
        units=1;
end

if nargin>3
    errordlg('Too much input arguments, please give only 3 arguments','Error');
end

if nargin==3
    if(~strcmp(varargin{1}, 'Threshold'))
    errordlg('The last argument must be ''Threshold'', if you want to use the threshold method', 'Error');
    return
    else
       %Executed when threshold method used
       for i=length(EventDatabase):-1:1
           leng=EventDatabase(i).StartAndEndPoint(2)-EventDatabase(i).StartAndEndPoint(1);
           leng=leng/SamplingFrequency*units;
           if (leng>numbers)
                RawSignal(EventDatabase(i).StartAndEndPoint(1):EventDatabase(i).StartAndEndPoint(2))=[];
           end
       end
    end  
end

if nargin==2
    numbers=sort(unique(numbers),'descend');
    for i=numbers
       RawSignal(EventDatabase(i).StartAndEndPoint(1):EventDatabase(i).StartAndEndPoint(2))=[];
    end  
end

newRawSignal=RawSignal;
end