function ARL=ARLSiegmund(deltax,sigmax,change,h)
%
% ARL=ARLSiegmund(deltax,sigmax,change,h);
%
% Siegmund's approximation of the ARL function
% for a change in the mean in the white gaussian case
%
%Inputs:
% - deltax: most likely change magnitude in the signal
% - sigmax: standard deviation of the signal
% - change: change value at which the ARL function is approximated
% - h: detection threshold(s) at which the ARL function is approximated
%
%Outputs:
% - ARL: approximation of the ARL function
%
%Interesting values for ARL function are obtained for:
% - change = 0
%  -> ARL_0 (one-sided cusum mean time between false detections)
% - change = +deltax
%  -> ARL_1 (one-sided cusum mean detection delay)
% - change = -deltax
%  -> ARL_{-1} (useful for two-sided cusum)


% mean and standard deviation of the inst. LR s[k]
mu=deltax*(change-deltax/2)/sigmax^2;
sigma=abs(deltax)/sigmax;

% normalized quantities
mun=mu/sigma;
hn=h/sigma;

% Siegmund's approximation of the ARL function
if (mu==0)
    ARL=(hn+1.166)^2;
else
    ARL=(exp(-2*mun*(hn+1.166))-1+2*mun*(hn+1.166))/(2*mun^2);
end
