function [] = penlabstresstest(feeder,timestats,logs)
% PENLABSTRESSTEST runs a series of tests on the solver.
%
% This is to be used as an automated tool to run all possible
% test of the solver (e.g., a whole benchmarking library) to check
% how a possible source change influenced the behaviour of
% the solver.
%
% The problems to test are generated by FEEDER. This routine executes them 
% and prints out one-liner statistics for comparison.
%
% INPUT:
%   feeder - routine returning the penlab objects to be ran,
%            optional, if not present, the default feeder is used;
%            it should look like:
%               function [status,name,prob,res] = feeder(no)
%            and to test number no=1..maxtests return status=0,
%            problem name and problem as penlab object,
%            expected results may be present in res, otherwise [],
%            if there are problems with the test, status<0,
%            status=1 means out of range of test
%   timestats - if (timestats), timing (tic/toc) is printed in the summary,
%            optional, default true
%   logs - if (logs), logs are produced for each of the tests in
%            the current directory; optional, default false
%
% EXAMPLES:
%   % use Matlab's diary to record the summary output
%   diary('logtest_summary.txt');
%   penlabstresstest();
%   diary off;
%
%   % using your own test set and default feeder:
%   mytestset = {{ 'sdpa', 'datafiles/truss1.dat-s', -8.999996e+00 }, ...
%                { 'bmi', 'datafiles/bmi_example.mat' }};
%   myfeeder = @(no) penlabtestfeeder(no,mytestset);
%   penlabstresstest(myfeeder);
%
%   % default feeder but turn off timing and generate logs
%   penlabstresstest(@penlabtestfeeder,0,1);
%
%
% See also penlabtestfeeder
%

% This file is a part of PENLAB package distributed under GPLv3 license
% Copyright (c) 2013 by  J. Fiala, M. Kocvara, M. Stingl
% Last Modified: 5/12/2013

  % relative precision for comparison the objective?
  tol = 1e-4;

  if (nargin<3)
    logs = 0;
  end

  if (nargin<2)
    timestats = 1;
  end

  if (nargin<1)
    % use default feeder with the default test set
    feeder = @penlabtestfeeder;
  end

  % run each test feeder returns
  no = 1;
  nowrong = 0;
  while (1)
    
    try
      [status,name,prob,res] = feeder(no);
    catch
      status = -99;
      name = 'FEEDERPROBLEM';
      prob = [];
      res = [];
    end

    if (status==1)
      % no more tests
      break;
    elseif (status==0)
      % all OK, run test

      try
        % set common option settings:
        % turn off printing to the screen
        prob.opts.outlev = 0;
        if (logs)
          c = clock;
          stamp = sprintf('%04i%02i%02i-%02i%02i',c(1:5));
          prob.opts.outlev_file = 2;
          prob.opts.out_filename = ['logtest-' name '-' stamp '.txt'];
        else
          prob.opts.outlev_file = 0;
        end

        ifail = prob.solve();

        if (~isempty(res))
          if (abs(prob.objx - res)<tol*(abs(prob.objx)+abs(res)))
            cmp = 'ok';
          else
            cmp = 'XX';
          end
        else
          cmp = '  ';
        end
        if (timestats)
          fprintf('%-20s: %10.1fs   %4i/%4i/%4i   %10.2e  %2s   %2i\n',name,...
            prob.stats_time_total,prob.miter,prob.initer,prob.lsiter,...
            prob.objx,cmp,ifail);
        else
          fprintf('%-20s: %4i/%4i/%4i   %10.2e  %2s   %2i\n',name,...
            prob.miter,prob.initer,prob.lsiter,...
            prob.objx,cmp,ifail);
        end
      catch
        fprintf('%-20s: SOLVER HARD ERROR!\n',name);
      end

      clear prob;

    else
      % problem with generating the test
      fprintf('%-20s: cannot generate (%i)\n',name,status);
      nowrong = nowrong + 1;
      if (nowrong>10)
        break;
      end
    end

    no = no + 1;

  end

  fprintf('END OF TESTS\n');

end

