function [penm] = bmi_define(bmidata)
% BMI_DEFINE defines penm structure for Bilinear Matrix Inequality SPD 
% created from user's bmidata structure. This is just a special
% version of PMI with assumed order 2 (or optionally 1 for linear
% SDP). Restricting the order helps a lot during evaluation of 
% matrix constraints and their derivatives. Note that structure
% 'bmidata' is fully compatible with PMI and thus can be used
% by pmi_define() as well.
%
% Typical way of invoking the solver:
%    penm = bmi_define(users_bmidata);
%    prob = penlab(penm);
%    prob.solve();
%    ...
%
% It is expected that 'bmidata' stores data for the following problem
%    min   c'x + 1/2 x'Hx
%    s.t.  lbg <= B*x <= ubg
%          lbx <=  x  <= ubx 
%          A_k(x)>=0     for k=1,..,Na
%    where
%          A_k(x) = A_0 + sum_i x_i*A_i + sum_ij x_i*x_j*K_ij
%    or written in the same notation as in PMI using multi-indices
%          A_k(x) = sum_i  x(multi-index(i))*Q_i
%    for example
%          A_k(x) = Q_1 + x_1*x_3*Q_2 + x_2*x_2*Q_3
%          thus multi-indices are  
%             midx_1 = 0       (absolute term, Q_1)
%             midx_2 = [1,3]   (bilinear/quadratic term, Q_2)
%             midx_3 = [2,2]   (bilinear/quadratic term, Q_3)
%
% List of elements of the user structure 'bmidata'
%   name ... [optional] name of the problem
%   Nx ..... number of primal variables
%   Na ..... [optional] number of matrix inequalities (or diagonal blocks
%            of the matrix constraint)
%   xinit .. [optional] dim (Nx,1), starting point
%
%   c ...... [optional] dim (Nx,1), coefficients of the linear obj. function,
%            considered a zero vector if not present
%   H ...... [optional] dim (Nx,Nx), Hessian for the obj. function,
%            considered a zero matrix if not present
%
%   lbx,ubx. [optional] dim (Nx,1) or scalars (1x1), lower and upper bound
%            defining the box constraints
%   B ...... [optional] dim(Ng,Nx), matrix defining the linear constraints
%   lbg,ubg. [optional] dim (Ng,1) or scalars, upper and lower bounds for B
%
%   A ...... if Na>0, cell array of A{k} for k=1,...,Na each defining 
%            one matrix constraint; let's assume that A{k} has maximal
%            order maxOrder=2 and has nMat matrices defined, then A{k} should 
%            have the following elements:
%              A{k}.Q - cell array of nMat (sparse) matricies of the same
%                 dimension
%              A{k}.midx - matrix maxOrder x nMat defining the multi-indices
%                 for each matrix Q; use 0 within the multi-index to reduce
%                 the order
%            for example, A{k}.Q{i} defines i-th matrix to which belongs
%            multi-index  A{k}.midx(:,i). If midx(:,i) = [1;3], it means
%            that Q_i is multiplied by x_1*x_3 within the sum;
%            midx(:,j)=[0;0], it means that Q_j is absolute term.
%
% See also yalmip2bmi

% This file is a part of PENLAB package distributed under GPLv3 license
% Copyright (c) 2013 by  J. Fiala, M. Kocvara, M. Stingl
% Last Modified: 27 Nov 2013

  penm = [];
  userdata = [];

  if (isfield(bmidata,'name'))
    penm.probname=bmidata.name;
  end
  penm.comment = 'Structure PENM generated by bmi_define()';

  Nx=bmidata.Nx;
  if (Nx>0)
    penm.Nx=Nx;
  else
    error('Input: Nx<=0');
  end
  
  % initial point
  if (isfield(bmidata,'xinit') && ~isempty(bmidata.xinit))  
    [n m] = size(bmidata.xinit);
    if (n==1 && m==Nx)
      penm.xinit = bmidata.xinit';
    elseif (n==Nx && m==1)
      penm.xinit = bmidata.xinit;
    else
      error('Input: xinit incompatible dimension.');
    end
  end

  % box constraints, dimensions will be checked inside Penlab
  if (isfield(bmidata,'lbx') && ~isempty(bmidata.lbx))
    penm.lbx=bmidata.lbx;
  end
  if (isfield(bmidata,'ubx') && ~isempty(bmidata.ubx))
    penm.ubx=bmidata.ubx;
  end

  % objective function
  if (isfield(bmidata,'c') && ~isempty(bmidata.c))
    [n m] = size(bmidata.c);
    if (n==1 && m==Nx)
      userdata.c=bmidata.c';
    elseif (n==Nx && m==1)
      userdata.c=bmidata.c;
    else
      error('Input: c incompatible dimension.');
    end
  else
    userdata.c=sparse(Nx,1);
  end
  if (isfield(bmidata,'H') && ~isempty(bmidata.H))
    [n m] = size(bmidata.H);
    if (n==Nx && m==Nx)
      userdata.H=bmidata.H;
    else
      error('Input: H incompatible dimensions.');
    end
  else
    userdata.H=sparse(Nx,Nx);
  end

  % linear constraints
  if (isfield(bmidata,'B') && ~isempty(bmidata.B))
    [m n] = size(bmidata.B);
    if (n~=Nx)
      error('Input: wrong dimension of B, should be Ng x Nx');
    end
    userdata.B=bmidata.B;
    Ng=m;
  else
    userdata.B=[];
    Ng=0;
  end
  penm.NgLIN=Ng;

  if (Ng>0)
    if (isfield(bmidata,'lbg') && ~isempty(bmidata.lbg))
      penm.lbg=bmidata.lbg;
    end
    if (isfield(bmidata,'ubg') && ~isempty(bmidata.ubg))
      penm.ubg=bmidata.ubg;
    end
  end

  % matrix constraints
  if (isfield(bmidata,'Na') && ~isempty(bmidata.Na) && bmidata.Na>0)
    % check every matrix constraint & detect if linear/nonlinear
    NA=bmidata.Na;
    userdata.NA=NA;
    NANLN=0;
    NALIN=0;
    list_nln=zeros(NA,1);
    list_lin=zeros(NA,1);
    
    for k=1:NA
      [userdata.mcon{k},order]=check_mcon(Nx,bmidata.A{k}.midx,bmidata.A{k}.Q);
      if (order==1)
        NALIN = NALIN+1;
        list_lin(NALIN) = k;
      else
        NANLN = NANLN+1;
        list_nln(NANLN) = k;
      end
    end

    % store the order, nonlinear first followed by linear
    userdata.mconorder = [list_nln(1:NANLN); list_lin(1:NALIN)];

    penm.NANLN=NANLN;
    penm.NALIN=NALIN;

    % let's make the constraints positive semidefinite (A_k(x)>=0)
    penm.lbA=zeros(NA,1);

  else
    penm.NANLN=0;
    penm.NALIN=0;
    userdata.NA=0;
  end

  % keep the proccessed structure
  penm.userdata=userdata;

  penm.objfun = @bmi_objfun;
  penm.objgrad = @bmi_objgrad;
  penm.objhess = @bmi_objhess;

  penm.confun = @bmi_confun;
  penm.congrad = @bmi_congrad;
  %penm.conhess = ...;  not needed because all linear

  penm.mconfun = @bmi_mconfun;
  penm.mcongrad = @bmi_mcongrad;
  % it is possible to choose how 2nd derivatives are included into
  % the Augmented Lagrangian; mconhess provides just one derivative
  % w.r.t. given i,j (d2/dxi dxj A_k(x)) and it is easy to write,
  % however, there is certain overhead of calling this routine repetitively
  % An alternative is that the whole contribution of all 2nd derivatives
  % of the given matrix constriant A_k(x) is computed on the user's side
  % which should be in general the faster approach.
  penm.mconhess = @bmi_mconhess;
  %penm.mconlagrhess = @bmi_mconlagrhess;

end



%%%%%%%%%%
% check one BMI matrix constraint
% input:
%   Nx - number of variables (to check midx)
%   midx(maxOrder,nMat) - multiindices for each matrix
%   Q{nMat} - cell array of nMat sparse matrices forming the matrix constraint
function [mcon,maxOrder] = check_mcon(Nx,midx,Q)

  % check that every matrix has a multi-index
  [maxOrder, nMat] = size(midx);
  [m, n] = size(Q);
  if (max(m,n) ~= nMat || min(m,n)~=1)
    error('Input: not matching number of matrices and their multi-indices');
  end
  if (nMat<1 || maxOrder<1)
    error('Input: empty matrix constraint');
  end
  if (maxOrder>2)
    error('Input: matrix constraint with higher order than 2, use PMI instead.');
  end

  % check multiindex (only elements 0..Nx are allowed)
  if (any(any(midx<0)) || any(any(midx>Nx)))
    error('Input: multiindex for matrix constraint is out of range 0..Nx.');
  end

  % check dimensions
  [dimQ, n]=size(Q{1});
  if (dimQ~=n)
    error('Input: not square matrix');
  end
  for i=2:nMat
    [m,n] = size(Q{i});
    if (m~=dimQ || n~=dimQ)
      error('Input: not matching dimensions within one matrix constraint');
    end
  end
  mcon.dim=dimQ;
  mcon.midx=midx;
  mcon.Q=cell(nMat,1);
  % copy & force sparsity
  for i=1:nMat
    mcon.Q{i}=sparse(Q{i});
  end

end

