function H = hermitesof(A,B,C,K,S) 
% Given a triplet (A,B,C) in state-space format
% and a feedback matrix K (YALMIP object SDPVAR)
% the instruction
%
%  H = hermitesof(A,B,C,K)
%
% returns the symmetric Hermite matrix H
% of the characteristic polynomial det(s*I-A-B*K*C)
%
% Matrix H depends polynomially on the entries of K,
% it is a YALMIP object of class SDPVAR
%
% Matrix H is positive definite if and only if
% A+B*K*C is Hurwitz stable
%
% Calls function HERMCOEF

% D. Henrion, 11 July 2007
% This file is a part of PENLAB package distributed under GPLv3 license
% Last Modified: 27 Nov 2013
  
n = size(A,1); m = size(B,2); p = size(C,1);
if size(K) ~= [m,p]
 error('Inconsistent dimensions')
end

if nargin < 5
 S = [];
end
if isempty(S)
 S = [0 1;1 0]; % continuous-time stability
end

% YALMIP decision variables
k = reshape(K,m*p,1); % in a column vector

% Build characteristic polynomial
s = sdpvar(1); % Laplace variable
tt = s*eye(n)-A-B*K*C;
ps = detn(tt);
[p,mons] = coefficients(ps,s);
d = degree(mons); % higest degree
if length(mons)+1 < d
 error('missing monomials')
end

% Build Hermite matrix
% lower tri coeffs
HC = hermcoef(S,d);

% bilin monom -> lower tri bilin monom
P = p*p'; pred = P(1,1);
for i=1:d
 pred = [pred; P(1:(i+1),i+1)];
end

% lower tri Hermite mat -> Hermite mat
Hred = HC*pred;
H = sdpvar(d,d);
i1 = 0; for i=1:d
 H(1:i,i) = Hred(i1+(1:i));
 H(i,1:i) = Hred(i1+(1:i))';
 i1 = i1+i;
end

if all(S(:)==[0;1;1;0])
 % for continuous-time stability
 % split odd and even part of Hermite matrix
 H = blkdiag(H(1:2:d,1:2:d),H(2:2:d,2:2:d));
end

