%CODEGENERATION.GENMEXINERTIA Generate C-MEX-function for robot inertia matrix
%
% cGen.genmexinertia() generates robot-specific MEX-functions to compute
% robot inertia matrix.
%
% Notes::
% - Is called by CodeGenerator.geninertia if cGen has active flag genmex
% - The MEX file uses the .c and .h files generated in the directory 
%   specified by the ccodepath property of the CodeGenerator object.
% - Access to generated functions is provided via subclass of SerialLink
%   whose class definition is stored in cGen.robjpath.
% - You will need a C compiler to use the generated MEX-functions. See the 
%   MATLAB documentation on how to setup the compiler in MATLAB. 
%   Nevertheless the basic C-MEX-code as such may be generated without a
%   compiler. In this case switch the cGen flag compilemex to false.
%
% Author::
%  Joern Malzahn, (joern.malzahn@tu-dortmund.de)
%
% See also CodeGenerator.CodeGenerator, CodeGenerator.geninertia.

% Copyright (C) 2012-2014, by Joern Malzahn
%
% This file is part of The Robotics Toolbox for Matlab (RTB).
%
% RTB is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% RTB is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU Lesser General Public License for more details.
%
% You should have received a copy of the GNU Leser General Public License
% along with RTB.  If not, see <http://www.gnu.org/licenses/>.
%
% http://www.petercorke.com
%
% The code generation module emerged during the work on a project funded by
% the German Research Foundation (DFG, BE1569/7-1). The authors gratefully
% acknowledge the financial support.

function [] = genmexinertia(CGen)

%% Individual inertia matrix rows
CGen.logmsg([datestr(now),'\tGenerating MEX-function for the robot inertia matrix row' ]);

Q = CGen.rob.gencoords;
nJoints = CGen.rob.n;

for kJoints = 1:nJoints
    CGen.logmsg(' %s ',num2str(kJoints));
    symname = ['inertia_row_',num2str(kJoints)];
    fname = fullfile(CGen.sympath,[symname,'.mat']);
    
    if exist(fname,'file')
        tmpStruct = load(fname);
    else
        error ('genmfuninertia:SymbolicsNotFound','Save symbolic expressions to disk first!')
    end
    
    funfilename = fullfile(CGen.robjpath,[symname,'.c']);
    
    % Function description header
    hStruct = createHeaderStructRow(CGen.rob,kJoints,symname);   %generate header
    
    % Generate and compile MEX function
    CGen.mexfunction(tmpStruct.(symname), ...
        'funfilename',funfilename,...
        'funname',[CGen.getrobfname,'_',symname],...
        'vars',{Q},...
        'output',['I_row',num2str(kJoints)],...
        'header',hStruct);
    
end
CGen.logmsg('\t%s\n',' done!');

%% Full inertia matrix
CGen.logmsg([datestr(now),'\tGenerating full inertia matrix m-function']);

symname = 'inertia';
f = sym(zeros(nJoints)); % dummy symbolic expression
funfilename = fullfile(CGen.robjpath,[symname,'.c']);

funname = [CGen.getrobfname,'_',symname];

hStruct = createHeaderStructFull(CGen.rob,symname); % create header
hFString = CGen.constructheaderstringc(hStruct);

% Generate and compile MEX function 
CGen.mexfunctionrowwise(f,...
    'funfilename',funfilename,...
    'funname',[CGen.getrobfname,'_',symname],...
    'vars',{Q},...
    'output','I',...
    'header',hStruct);
end

%% Definition of the header contents for each generated file
function hStruct = createHeaderStructRow(rob,curJointIdx,fName)
[~,hStruct.funName] = fileparts(fName);
hStruct.calls = '';
hStruct.shortDescription = ['Computation of the robot specific inertia matrix row for corresponding to joint ', num2str(curJointIdx), ' of ',num2str(rob.n),'.'];
hStruct.detailedDescription = {'Given a full set of joint variables this function computes the',...
    ['inertia matrix row number ', num2str(curJointIdx),' of ',num2str(rob.n),' for ',rob.name,'. Angles have to be given in radians!']};
hStruct.inputs = {['input1:  ',int2str(rob.n),'-element vector of generalized coordinates.']};
hStruct.outputs = {['I_row_',int2str(curJointIdx),':  [1x',int2str(rob.n),'] output row of the robot inertia matrix.']};
hStruct.references = {'Robot Modeling and Control - Spong, Hutchinson, Vidyasagar',...
    'Modelling and Control of Robot Manipulators - Sciavicco, Siciliano',...
    'Introduction to Robotics, Mechanics and Control - Craig',...
    'Modeling, Identification & Control of Robots - Khalil & Dombre'};
hStruct.authors = {'This is an autogenerated function!',...
    'Code generator written by:',...
    'Joern Malzahn (joern.malzahn@tu-dortmund.de)'};
hStruct.seeAlso = {'coriolis'};
end

function hStruct = createHeaderStructFull(rob,fname)
[~,hStruct.funName] = fileparts(fname);
hStruct.calls = '';
hStruct.shortDescription = ['Inertia matrix for the ',rob.name,' arm.'];
hStruct.detailedDescription = {'Given a full set of joint variables the function computes the',...
    'inertia Matrix of the robot. Angles have to be given in radians!'};
hStruct.inputs = {['input1:  ',int2str(rob.n),'-element vector of generalized coordinates.']};
hStruct.outputs = {['I:  [',int2str(rob.n),'x',int2str(rob.n),']output inertia matrix.']};
hStruct.references = {'Robot Modeling and Control - Spong, Hutchinson, Vidyasagar',...
    'Modelling and Control of Robot Manipulators - Sciavicco, Siciliano',...
    'Introduction to Robotics, Mechanics and Control - Craig',...
    'Modeling, Identification & Control of Robots - Khalil & Dombre'};
hStruct.authors = {'This is an autogenerated function!',...
    'Code generator written by:',...
    'Joern Malzahn (joern.malzahn@tu-dortmund.de)'};
hStruct.seeAlso = {'coriolis'};
end