function s = test_lwpr_1D(s)

  global lwprs;

  % load training and testing data
  load train.data;
  [dum,inds]=sort(train(:,1));
  train = train(inds,:);
  X=train(:,1);
  Y=train(:,2);
  n = length(X);
  
  load test.data;
  [dum,inds]=sort(test(:,1));
  test = test(inds,:);
  Xt=test(:,1);
  Yt=test(:,2);
  
  norm_in = range(X)/2;
  norm_out = range(Y)/2;

  % initialize LWPR
  ID = 1;
  if ~exist('s') | isempty(s)
    lwpr('Init',ID,1,1,0,0,0,1e-8,50,norm_in,norm_out,'lwpr_test');
  else
    lwpr('Init',ID,s);
  end

  % set some parameters
  kernel = 'Gaussian';
  % kernel = 'BiSquare'; % note: the BiSquare kernel requires different values for
  %                              an initial distance metric, as in the next line
  % lwpr('Change',ID,'init_D',[7 0.01; 0.01 7]);

  lwpr('Change',ID,'init_D',30); 
  lwpr('Change',ID,'init_alpha',100); 
  lwpr('Change',ID,'w_gen',0.5);             % more overlap gives smoother surfaces
  lwpr('Change',ID,'meta',1);                % meta learning can be faster, but numerical more dangerous
  lwpr('Change',ID,'meta_rate',100);
  lwpr('Change',ID,'init_lambda',0.995);
  lwpr('Change',ID,'final_lambda',0.9999);
  lwpr('Change',ID,'tau_lambda',0.99999);

  % train the model
  for j=1:100
    inds = randperm(n);
    mse = 0;
    for i=1:n,
      [yp,w] = lwpr('Update',ID,X(inds(i),:)',Y(inds(i),:)');
      mse = mse + (Y(inds(i),:)-yp).^2;
    end
    nMSE = mse/n/var(Y,1);	
    disp(sprintf('#Data=%d #rfs=%d nMSE=%5.3f (TrainingSet)',lwprs(ID).n_data,length(lwprs(ID).rfs),nMSE));
  end

  % create predictions for the test data
  Yp   = zeros(size(Yt));
  Conf = zeros(size(Yt));
  for i=1:length(Xt),
    [yp,w,conf]=lwpr('Predict',ID,Xt(i,:)',0.001);
    Yp(i,1) = yp;
    Conf(i,1)=conf;
  end
  ep   = Yt-Yp;
  mse  = mean(ep.^2);
  nmse = mse/var(Y,1);
  disp(sprintf('#Data=%d #rfs=%d nMSE=%5.3f (TestSet)',lwprs(ID).n_data,length(lwprs(ID).rfs),nmse));

  % get the data structure
  s = lwpr('Structure',ID);

  figure(1);
  clf;

  % plot the raw noisy data
  subplot(2,1,1);
  plot(Xt,Yt,'k',X(:,1),Y,'*',Xt,Yp,'r',Xt,Yp+Conf,'r:',Xt,Yp-Conf,'r:');
  title(sprintf('Noisy data samples and fitted function: nMSE=%5.3f',nmse));
  
  % plot the local models
  subplot(2,1,2);
  xmin=min(Xt);
  xmax=max(Xt);
  dx=range(Xt)/200;
  Xk=xmin:dx:xmax;
  for i=1:length(lwprs(ID).rfs),
    N = 1/norm_in;
    Yk=exp(-0.5*N*lwprs(ID).rfs(i).D*N*(Xk-lwprs(ID).rfs(i).c.*norm_in).^2);
    plot(Xk,Yk,'k');
    hold on;
  end
  hold off;
  title('RF Kernels');
  
  save results.mat Xt Yp Conf -mat

% --------------------------------------------------------------------------------
function [X,Y,Z]=makesurf(data,nx)
% [X,Y,Z]=makesurf(data,nx) converts the 3D data file data into
% three matices as need by surf(). nx tells how long the row of the
% output matrices are

  [m,n]=size(data);

  n=0;
  for i=1:nx:m,
    n = n+1;
    X(:,n) = data(i:i+nx-1,1);
    Y(:,n) = data(i:i+nx-1,2);
    Z(:,n) = data(i:i+nx-1,3);
  end;


% --------------------------------------------------------------------------------
function []=draw_ellipse(M,C,w,kernel)
% function draw ellipse draws the ellipse corresponding to the
% eigenvalues of M at the location c.

  [V,E] = eig(M);

  E = E;
  d1 = E(1,1);
  d2 = E(2,2);

  steps = 50;
  switch kernel
   case 'Gaussian'
    start = sqrt(-2*log(w)/d1);
   case 'BiSquare'
    start = sqrt(2*(1-sqrt(w))/d1);
  end


  for i=0:steps,
    Xp(i+1,1) = -start + i*(2*start)/steps;
    switch kernel
     case 'Gaussian'
      arg = (-2*log(w)-Xp(i+1,1)^2*d1)/d2;
     case 'BiSquare'
      arg = (2*(1-sqrt(w))-Xp(i+1,1)^2*d1)/d2;
    end
    if (arg < 0), 
      arg = 0; 
    end; % should be numerical error
    Yp(i+1,1) = sqrt(arg);
  end;

  for i=1:steps+1;
    Xp(steps+1+i,1) = Xp(steps+1-i+1,1);
    Yp(steps+1+i,1) = -Yp(steps+1-i+1,1);
  end;

  % tranform the rf

  M = [Xp,Yp]*V(1:2,1:2)';

  Xp = M(:,1) + C(1);
  Yp = M(:,2) + C(2);

  plot(C(1),C(2),'ro',Xp,Yp,'c');

