%CODEGENERATOR.GENMATVECPRODC Generates a matrix-vector product C-implementation.
%
% cGen.gendotprodc generates a .h and a .c file in the directory
% specified by ccodepath.
%
% Notes::
% - Is called by geninvdyn and genfdyn if cGen has active flag genmex or genccode.
%
% Authors::
%  Joern Malzahn   (joern.malzahn@tu-dortmund.de) 
%
% See also CodeGenerator, CodeGenerator.gengaussjordanc.

% Copyright (C) 1993-2014, by Peter I. Corke
% Copyright (C) 2014, by Joern Malzahn
%
% This file is part of The Robotics Toolbox for Matlab (RTB).
%
% RTB is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% RTB is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU Lesser General Public License for more details.
%
% You should have received a copy of the GNU Lesser General Public License
% along with RTB.  If not, see <http://www.gnu.org/licenses/>.
%
% http://www.petercorke.com
function [] =  genmatvecprodc(CGen)

funname = 'matvecprod';
funfilename = [funname,'.c'];
hfilename = [funname,'.h'];
srcDir = fullfile(CGen.ccodepath,'src');
hdrDir = fullfile(CGen.ccodepath,'include');

% Check for existance of dotProd C-files
if exist(fullfile(srcDir,funfilename),'file') && exist(fullfile(srcDir,funfilename),'file')
    return;
end

% Check for existance of C-code directories
if ~exist(srcDir,'dir')
    mkdir(srcDir);
end
if ~exist(hdrDir,'dir')
    mkdir(hdrDir);
end

% Create the function description header
hStruct = createHeaderStruct(funname); % create header
if ~isempty(hStruct)
    hFString = CGen.constructheaderstringc(hStruct);
end

%% Generate C implementation file
fid = fopen(fullfile(srcDir,funfilename),'w+');

% Function description header
fprintf(fid,'%s\n\n',hFString);

% Includes
fprintf(fid,'%s\n\n',...
    ['#include "', hfilename,'"']);

% Function
fprintf(fid,'%s\n',['void ', funname, '(double *outVector, const double *inMatrix, const double *inVector, int nRow, int nCol){']);
fprintf(fid,'%s\n',' '); % empty line    
fprintf(fid,'\t%s\n','int iRow, iCol = 0;');
fprintf(fid,'%s\n',' '); % empty line    
fprintf(fid,'\t%s\n','for (iRow = 0; iRow < nRow; iRow++){');
fprintf(fid,'\t\t%s\n','for (iCol = 0; iCol < nCol; iCol++){');
fprintf(fid,'\t\t\t%s\n','outVector[iCol] += inMatrix[nRow*iRow+iCol] * inVector[iRow];');
fprintf(fid,'\t\t%s\n','}');
fprintf(fid,'\t%s\n','}  ');
fprintf(fid,'%s\n','}');

fclose(fid);

%% Generate C header file
fid = fopen(fullfile(hdrDir,hfilename),'w+');

% Function description header
fprintf(fid,'%s\n\n',hFString);

% Include guard
fprintf(fid,'%s\n%s\n\n',...
    ['#ifndef ', upper([funname,'_h'])],...
    ['#define ', upper([funname,'_h'])]);

% Function prototype
fprintf(fid,'%s\n\n',['void ',funname,'(double *outVector, const double *inMatrix, const double *inVector, int nRow, int nCol);']);

% Include guard
fprintf(fid,'%s\n',...
    ['#endif /*', upper([funname,'_h */'])]);

fclose(fid);

CGen.logmsg('\t%s\n',' done!');

end

 %% Definition of the function description header contents
function hStruct = createHeaderStruct(fname)
[~,hStruct.funName] = fileparts(fname);
hStruct.shortDescription = ['Compute the product of a matrix and a vector'];
hStruct.calls = ['void ',hStruct.funName,'(double *outVector, const double *inMatrix, const double *inVector, int nRow, int nCol)'];
hStruct.detailedDescription = {['Given an [nRow x nCol] inMatrix and a nCol-element inVector, the function'],...
    'computes the nRow-element outVector as: outVector = inMatrix*inVector.'};
hStruct.inputs = { ['inMatrix: [nRow x nCol] input matrix,'],...
                   ['inVector: nCol-element input vector,'],...
                   ['nRow: number of rows of inMatrix,'],...
                    'nCol: number of columns of inMatrix and elements of inVector.'};
hStruct.outputs = {['outVector:  nRow-element output vector.']};
hStruct.references = {'The C Book: http://publications.gbdirect.co.uk/c_book/'};
hStruct.authors = {'This is an autogenerated function!',...
    'Code generator written by:',...
    'Joern Malzahn (joern.malzahn@tu-dortmund.de)'};
hStruct.seeAlso = {'CodeGenerator'};
end