#pragma rtGlobals=1		// Use modern global access method.

//
// Functions
// =======
// a) Creating input data files for the fortran functions 
//	output_model_data(opt,run_list,flag_name)
//
// b) After model calculations
//	load_res_file(io_opt,refNum)
//		Function reads selected *.RES file and optionally puts the results to corresponding waves
//
//	res2wave_growth(reset,ID_name,data_name,just_daq)
//		Function reads results from droplet growth calculations (old IO format)
//
// **********************************************************************************************************************



function output_model_data(opt,run_list,flag_name)
	// Function for generating input data files for the droplet growth model and alpha solver
	// New (8/5/2011) free format, where output depends on input parameters
	variable opt			// Case option
	string run_list		// Optional comma separated list of selected scan IDs and intervals
	string flag_name		// Optional flag wave name (non-zero=good data)
	//
	variable alpha_calc=0	// Either droplet growth or alpha solver
	//
	// The data for generating outputs
	// =======================
	// 1) Scan ID wave: an integer index wave
	wave Scan_ID	// Always wave
	// 2) Instrument settings: P, Q, SS (or Thot), SAR, Tinlet and Tcool
	wave Q_wave, P_wave, SS_wave, SAR_wave, Tinlet_wave, Tcool_wave // Either waves
	//	... or constant values
	variable Q_val=-1		// Flow rate (L/min)
	variable P_val=-1		// Pressure (mbar)
	variable SS_val=-1	// Instrument supersaturation (%)
	variable SAR_val=-1	// Sheath-to-aerosol flow ratio
	variable Tinlet_val=-1	// Inlet (sample and sheath flow) temperature (K)
	variable Tcool_val=-1	// Minimum (column top or T1) temperature (K)
	
	// 3) Kappa, alpha, dry size and number concentration (matrix or vector) waves
	wave kappa_wave, alpha_wave, dN_wave, dp_wave
	//   	... or constant values
	variable kappa_val=-1// Kappa (dimensionless)
	variable alpha_val=-1	// Alpha (dimensionless)
	variable dp_dry=-1	// Dry size (nm)
	variable dN_dry=-1	// Dry number concentration (#/cc)
	//	... or dry particle sizes can be loaded from an external text file
	string distr_name=""
	//	 dN/dlogDp can be converted to dN by using known bin widths
	variable convert_dn=0	// Default is to have dN as an input
	wave imdp, dlogDp	// Bin intervals (1) or dlogDp (2)
	// 4) Droplet size waves (alpha calculations only)
	wave CCN_wave,Davg_wave,Dsig_wave
	// ... or just statistical mean values
	variable CCN_val=-1	// CCN concentration (#/cc)
	variable Davg_val=-1	// Mean droplet diameter (um)
	variable Dsig_val=-1	// Standard deviation (um)
	//
	// a) Output format for droplet growth: [instrument settings], [particle properties]
	//	Instrument settings: Q, P, SS, SAR, Tinlet, Tcool
	//	Particle properties: Nk, Na, Nn, Nd, Nk x kappa, Na x alpha, Nn x Ni, Nd x Di
	//		Dimensions (Nk, Na, Nn, Nd) are detected automatically. Possible inputs are 
	//		constant (N=1), vectors (N=1) and/or matrixes (N>1).
	//
	// b) Output format for alpha calculation: [instrument settings], [particle properties], [droplet size]
	//	Instrument settings: same as above
	//	Particle properties (without alpha): Nk, Nn, Nd, Nk x kappa, Nn x Ni, Nd x Di
	//	Droplet size: CCN, Daq_avg, Daq_std
	//
	//		
	if (opt==0)
		// Sample calibarion experiment
		//
		// 1) Scan ID wave
		wave Scan_ID
		// 2) Instrument settings as P, Q , SS, SAR, Tcool and dTinlet waves
		wave Q_wave		// This wave exists, so this commad is not really needed (L/min)
		wave P_wave		// -||- (mbar)
		wave SS_wave=SS_cal	// The example supersaturation wave is called SS_cal (%)
		wave SAR_wave=SAR		// ... also the SAR wave is just called SAR
		wave Tinlet_wave=T_TEC1	// Inlet and column top temperatures should be equal (K)
		wave Tcool_wave=T_TEC1	// (K)
		// 3) Kappa and alpha: possible to use default values
		kappa_val=0.6	// Constant kappa for the calibration experiments
		alpha_val=0.2	// -||- alpha -||-
		// 4) Dry size distribution
		wave dN_wave=CCN_conc		// Here CCN concentration is used as is, but this could also be from CPC
		wave dp_wave=DMA_Diameter	// Scanning mobility, so dry size is varying
	else
		print "Wrong option?"
		return 0
	endif
	//
	// Scan ID must exist
	if (waveexists(Scan_ID)==0)
		print "Wave Scan_ID not found!"
		return 0
	endif	
	//
	// Optional flag/mask wave (non-zero=good data)
	if (strlen(flag_name))
		if (waveexists($flag_name)==0)
			print "Error: flag wave not found!"
			return 0
		endif
		wave flag_wave=$flag_name
	endif	
	//
	// Optional run list
	if (strlen(run_list)==0)
		// Save all if an empty run list is given
		run_list=num2str(Scan_ID[0])+"-"+num2str(Scan_ID[numpnts(Scan_ID)-1])
	endif
	//
	// Dimensions
	variable Nk=1, Na=1, Nd=1, Nn=1
	// a) kappa
	if (kappa_val>=0 || dimsize(kappa_wave,1)==0)// Scalar
	elseif (dimsize(kappa_wave,1))				// Matrix
		Nk=dimsize(kappa_wave,1)
	else
		print "Error: kappa data not found!"
		return 0
	endif
	// b) alpha or droplet size
	if (alpha_calc)
		// Alpha is calculated from droplet size distribution data
		if (CCN_val<=0 && waveexists(CCN_wave)==0)
			print "Error: CCN concentration wave not found!"
			return 0
		endif
		if (Davg_val<=0 && waveexists(Davg_wave)==0)
			print "Error: droplet size wave not found!"
			return 0
		endif
		if (Dsig_val<=0 && waveexists(Dsig_wave)==0)
			print "Error: droplet size STD wave not found!"
			return 0
		endif
	else
		// Alpha is one input
		if (alpha_val>=0 || dimsize(alpha_wave,1)==0)	// Scalar (or alpha is calculated)
		elseif (dimsize(alpha_wave,1))				// Matrix
			Na=dimsize(alpha_wave,1)	
		else
			print "Error: alpha data not found!"
			return 0		
		endif
	endif
	// c) Dry size
	if (strlen(distr_name))	// Saved to another text file
		Nd=0
	elseif (dp_dry>=0 || dimsize(dp_wave,1)==0)	// Scalar
	elseif (dimsize(dp_wave,1))				// Matrix
		Nd=dimsize(dp_wave,1)	
	else
		print "Error: particle size data not found!"
		return 0
	endif	
	// d) Concentration
	if (dN_dry>=0 || dimsize(dN_wave,1)==0)	// Scalar
	elseif (dimsize(dN_wave,1))
		Nn=dimsize(dN_wave,1)
	else
		print "Error: particle number concentration data not found!"
		return 0
	endif
	//
	// Open file
	variable refNum=-1
	if (alpha_calc)
		Open/D/T=".inp" refNum
	else
		Open/D/T=".ion" refNum
	endif
	if (strlen(S_fileName)==0)
		return 0
	endif
	if (refNum==-1)
		Open refNum as S_fileName
	endif
	//
	// Write header line
	string hdr="Scan ID	Q (L/min)	P (mbar)	SS (%)   	SAR      	Tinlet (K)	Tcool (K)"
	if (alpha_calc)
		hdr+="	Nkappa	Ndn	Ndp"
		hdr+="	[Nkappa x kappa]	[Nn x Ni], [Ndp x Dp]	[CCN, Daq, STD] \r"
	else
		hdr+="	Nkappa	Nalpha	Ndn	Ndp"
		hdr+="	[Nkappa x kappa]	[Nalpha x alpha]	[Nn x Ni], [Ndp x Dp] \r"
	endif
	fprintf refNum, "%s", hdr
	//
	string list
	variable run, run_first, run_last, ind
	variable p1, p2, p3, p4, p5, p6, p7
	variable n_flagged=0, n_printed=0
	variable i, j
	for (i=0;i<itemsinlist(run_list,",");i+=1)
		list=stringfromlist(i,run_list,",")
		if (itemsinlist(list,"-")==2)
			// Run interval
			run_first=str2num(stringfromlist(0,list,"-"))
			run_last=str2num(stringfromlist(1,list,"-"))
		else
			// Single run
			run_first=str2num(list)
			run_last=run_first
		endif
		//
		for (run=run_first;run<=run_last;run+=1)
			FindValue/T=(0.1)/V=(run) Scan_ID
			if (V_value==-1)
				print "Error: scan ID "+num2str(run)+" not found!"
				continue
			endif
			ind=V_value
			//
			if (strlen(flag_name) && (numtype(flag_wave[ind]) || flag_wave[ind]==0))
				// flag_wave: non-zero is good data
				n_flagged+=1
				continue
			endif
			//
			//
			//
			if (Q_val>0)
				p1=Q_val
			else
				if (n_printed==0 && waveexists(Q_wave)==0)
					print "Q wave not found!"
					break
				endif
				p1=Q_wave[ind]
			endif
			if (P_val>0)
				p2=P_val
			else
				if (n_printed==0 && waveexists(P_wave)==0)
					print "P wave not found!"
					break
				endif
				p2=P_wave[ind]
			endif
			if (SS_val>=0)
				p3=SS_val
			else
				if (n_printed==0 && waveexists(SS_wave)==0)
					print "SS wave not found!"
					break
				endif
				p3=SS_wave[ind]
			endif
 			if (SAR_val>0)
				p4=SAR_val
			else
				if (n_printed==0 && waveexists(SAR_wave)==0)
					print "SAR wave not found!"
					break
				endif
				p4=SAR_wave[ind]
			endif
			 if (Tinlet_val>0)
				p5=Tinlet_val
			else
				if (n_printed==0 && waveexists(Tinlet_wave)==0)
					print "Tinlet wave not found!"
					break
				endif
				p5=Tinlet_wave[ind]
			endif
			 if (Tcool_val>0)
				p6=Tcool_val
			else
				if (n_printed==0 && waveexists(Tcool_wave)==0)
					print "Tcool wave not found!"
					break
				endif
				p6=Tcool_wave[ind]
			endif
			//
			if (numtype(p1+p2+p3+p4+p5+p6))
				print "Warning: NaN data ignored for row "+num2str(ind)
				continue
			endif
			//
			// a) Instrument parameters: Scan ID, Q, P, SS, SAR, Tinlet and Tcool
			fprintf refNum, "\n%u\t%f\t%f\t%f\t%f\t%f\t%f\t", Scan_ID[ind], p1, p2, p3, p4, p5, p6
			//
			// b) Dimensions: Nk, Na, Nn, Nd
			if (alpha_calc)
				fprintf refNum, "%u\t%u\t%u", Nk, Nn, Nd
			else
				fprintf refNum, "%u\t%u\t%u\t%u", Nk, Na, Nn, Nd
			endif
			//
			// c) Particle properties: Nk x kappa, Na x alpha, Ns x Ni, Ns x Di
			// Kappa
			if (kappa_val>=0)
				// Scalar
				fprintf refNum, "\t%f", kappa_val
			elseif (dimsize(kappa_wave,1)==0)
				// Column vector
				if (numtype(kappa_wave[ind]))
					print "Error: NaN kappa!"
					break
				endif
				fprintf refNum, "\t%f", kappa_wave[ind]
			elseif (dimsize(kappa_wave,1))
				// Matrix
				if (numtype(kappa_wave[ind][0]))
					print "Error: NaN kappa!"
					break
				endif
				// Precision for the matrix inputs: 5 significant numbers (and exponent if needed)
				for (j=0;j<Nk;j+=1)
					fprintf refNum, "\t%.5g", kappa_wave[ind][j]
				endfor
			endif
			//
			// Alpha
			if (alpha_calc)
				// Alpha will be calculated
			elseif (alpha_val>=0)
				// Scalar
				fprintf refNum, "\t%f", alpha_val
			elseif (dimsize(alpha_wave,1)==0)
				// Column vector
				if (numtype(alpha_wave[ind]))
					print "Error: NaN alpha!"
					break
				endif				
				fprintf refNum, "\t%f", alpha_wave[ind]
			elseif (dimsize(alpha_wave,1))
				// Matrix
				if (numtype(alpha_wave[ind][0]))
					print "Error: NaN alpha!"
					break
				endif
				for (j=0;j<Na;j+=1)
					fprintf refNum, "\t%.5g", alpha_wave[ind][j]
				endfor
			endif
			//
			// Particle number
			if (dN_dry>=0)
				// Scalar
				fprintf refNum, "\t%f", dN_dry
			elseif (dimsize(dN_wave,1)==0)
				// Column vector
				if (numtype(dN_wave[ind]))
					print "Error: NaN number concentration!"
					break
				endif
				fprintf refNum, "\t%f", dN_wave[ind]
			elseif (dimsize(dN_wave,1))
				// Matrix, dN/dlogDp converted to dn
				if (numtype(dN_wave[ind][0]))
					print "Error: NaN number concentration!"
					break
				endif
				for (j=0;j<Nn;j+=1)
					if (convert_dn==1)
						// 1=Bin intervals: imdp
						fprintf refNum, "\t%.5g", dN_wave[ind][j]*log(imdp[j+1]/imdp[j])
					elseif (convert_dn)
						// 2=dlogdp wave
						fprintf refNum, "\t%.5g", dN_wave[ind][j]*dlogdp[j]
					else
						fprintf refNum, "\t%.5g", dN_wave[ind][j]
					endif
				endfor
			endif
			//
			// Dry particle sizes
			if (strlen(distr_name))
				// Distribution name given
				if (n_printed)
					fprintf refNum, "\t\"%s\"", ""
				else
					// Save the name only once
					fprintf refNum, "\t\"%s\"", distr_name
				endif				
			elseif (dp_dry>0)
				// Scalar
				fprintf refNum, "\t%f", dp_dry
			elseif (dimsize(dp_wave,1)==0)
				// Column vector
				if (numtype(dp_wave[ind]))
					print "Error: NaN dry particle size!"
					break
				endif				
				fprintf refNum, "\t%f", dp_wave[ind]
			elseif (dimsize(dp_wave,1))
				// Matrix
				if (numtype(dp_wave[ind][0]))
					print "Error: NaN dry particle size!"
					break
				endif
				for (j=0;j<Nd;j+=1)	
					fprintf refNum, "\t%.5g", dp_wave[ind][j]
				endfor
			endif
			//
			// d) Droplet size distribution (alpha calculations only)
			if (alpha_calc)
				if (CCN_val>=0)
					p1=CCN_val
				else
					p1=CCN_wave[ind]
				endif
				if (Davg_val>=0)
					p2=Davg_val
				else
					p2=Davg_wave[ind]
				endif
				if (Dsig_val>=0)
					p3=Dsig_val
				else
					p3=Dsig_wave[ind]
				endif
				fprintf refNum, "\t%f\t%f\t%f", p1, p2, p3
			endif
			//
			// End of line
			fprintf refNum, "\t\r"
			//
			n_printed+=1
		endfor
	endfor
	fprintf refNum, "\n"
	if (n_flagged)
		print "Flagged runs: "+num2str(n_flagged)+", good runs: "+num2str(n_printed)
	endif
	Close refNum
end







//
// **********************************************************************************************************************
// 
// Loading output files from fortran model calculations
//	1) load file
//		load_res_file(io_opt,refNum)
//	2) Extract the waves from the data matrix for growth
//		res2wave_growth(reset,ID_name,data_name,just_daq)
//

function load_res_file(io_opt,refNum)
	// Function for reading single file to wave tmp_data
	variable io_opt	// Different ways to read the data
	variable refNum 	// 0=ask which file to read, other=read a selected file
	//
	// Open file if refNum is not connected to any file
	if (refNum<=0)
		Open/D/R/T=".RES" refNum
		if (strlen(S_fileName)==0)
			return 0
		endif
		Open/R/T=".RES" refNum S_fileName
	endif
	//	
	variable test_items_per=0
	variable i, j, k, m, len, n=500, max_cols=1800, cols
	make/o/n=(n,max_cols) tmp_data
	// Header
	string line, tmp
	FReadLine refNum, line
	for (i=0;i<100000;i+=1)	// Max 10 000 lines
		// Note! There can not be more than 1000 characters per line. If this maximum
		// is exceeded, the data will be in more than one line.
		if (io_opt==0)
			// Just read one single line
			FReadLine/N=(8192) refNum, line
		elseif (io_opt<0)
			// Read until a "short line" is found
			line=""
			do
				FReadLine/N=(8192) refNum, tmp
				line+=tmp+" "
			while( strlen(tmp)>980 )				
		elseif (io_opt<=100)
			// Read fixed amount of lines (<100)
			line=""
			for (j=1;j<=io_opt;j+=1)
				FReadLine/N=(8192) refNum, tmp
				line+=tmp+" "
			endfor
		elseif (io_opt<=10000)
			// Read fixed amount of characters (<10 000)
			line=""
			j=0
			do
				FReadLine/N=(8192) refNum, tmp
				line+=tmp+" "
				j+=1 // Number of spaces
			while( strlen(line)<io_opt+j )			
		else
			// Count the numbers in the first line and then use this number to
			// determine the number of items in the following lines
			line=""
			len=0	// Number of items
			do
				FReadLine/N=(8192) refNum, tmp
				line+=tmp+" "
				//
				// Count the items
				if (i==0)
					// Find the total number of 
					test_items_per+=count_string_nums(tmp)
				else
					len+=count_string_nums(tmp)
					if (len>=test_items_per)
						break
					endif
				endif
			while( strlen(tmp)>980 )
		endif
		//
		len=strlen(line)
		if (len<10)	// End of file
			break
		endif
		//
		if (i==n)
			n+=1000
			redimension/n=(n,-1) tmp_data
		endif
		//
		if (1)
			// First non blanck character
			k=0		// Start of number
			j=0		// Current point, max value=len
			m=0	// Number of points
			do
				// Next non-blanck character
				j=j-1
				do	// ...while space
					j+=1
				while (j<len && cmpstr(line[j,j]," ")==0)
				if (j==len)	// Reached the end
					break
				endif
				k=j	// k is not a space
				//
				// Next blanck character
				do	// ...while not a space
					j+=1
				while (j<len && cmpstr(line[j,j]," ")!=0)
				// Convert to number
				tmp_data[i][m]=str2num(line[k,j])
				m+=1
			while (j<len && m<max_cols)
			//
			if (i==0)
				if (m==max_cols)
					print "Error: maximum number of columns ("+num2str(max_cols)+") found!"
					break
				endif
				cols=m
			elseif (m!=cols)
				print m, cols
				print "Error: different number of columns in row "+num2str(i+1)+"!"
				print line
				i+=1
				break
			endif
		else
			k=0
			m=0
			for (j=1;j<len;j+=1)
				if (cmpstr(line[j,j]," ")==0 || j==len-1)
					tmp_data[i][m]=str2num(line[k,j])
					m=m+1
					k=j
				endif
				if (m==57)
					break
				endif
			endfor
		endif
	endfor
	redimension/n=(i,cols) tmp_data
	//
	// Close file
	close(refNum)
	edit/K=1 tmp_data
	//
	print "File "+S_fileName+" loaded"
end
//
function count_string_nums(str)
	string str
	//
	variable len=strlen(str)
	if (len==0)
		return 0
	endif
	//
	variable n=0	// Number of points
	variable k=0		// Start of number
	variable j	=-1	// Current point, max value=len
	do
		// Next non-blanck character
		do	// ...while space
			j+=1
		while (j<len && cmpstr(str[j,j]," ")==0)
		if (j==len)	// Reached the end
			break
		endif
		// Next blanck character
		do	// ...while not a space
			j+=1
		while (j<len && cmpstr(str[j,j]," ")!=0)
		n+=1	// This was a number
		// Point j is blanck
	while (j<len)
	return n
end






function res2wave_growth(reset,ID_name,data_name,just_daq)
	variable reset
	string ID_name
	string data_name
	variable just_daq
	//
	// The data
	if (strlen(data_name)==0)
		data_name="tmp_data"
	endif
	if (waveexists($data_name)==0)
		print "Error: wave "+data_name+" not found!"
		return 0
	endif
	wave tmp_data=$data_name
	variable n=dimsize(tmp_data,0), n_cols=dimsize(tmp_data,1)
	//
	// Index wave
	if (strlen(ID_name)==0)
		ID_name="Scan_ID"
	endif
	if (waveexists($ID_name)==0)
		print "Error: wave "+ID_name+" not found!"
		return 0
	endif
	wave Scan_ID=$ID_name
	variable m=numpnts(Scan_ID)
	//
	//
	// New format
	// a) ID, So, S, Sp, Tmax, Tmin, Tin, P, Q  SAR, 
	// 	CCN, Davg, STD, N, N x DN, N x DAQ, NOPC, NOPC x DN
	// b) ID, Tmaxo, S, Sp, Tmax, Tmin, Tin, P, Q  SAR, 
	// 	CCN, Davg, STD, N, N x DN, N x DAQ, NOPC, NOPC x DN
	//
	// Matrix dimensions
	variable n_daq=tmp_data[0][13], n_opc=0, n_instr
	if (n_daq==round(n_daq))
		// Cases a and b
		n_instr=9	// So, S, Sp, Tmax, Tmin, Tin, P, Q  SAR,
		if (n_cols==14+2*n_daq)
			// No other data (OPC)
		elseif (tmp_data[0][14+2*n_daq]==round(tmp_data[0][14+2*n_daq]))
			n_opc=tmp_data[0][14+2*n_daq]
		else
			print "Unknown data format!"
			return 0
		endif
	else
		print "Unknown data format!"
		return 0
	endif
	//
	string wlist="edit/K=1 Scan_ID,sim_daq_avg"	
	// Use existing waves or reset
	if (waveexists(sim_io_mat)==0 || reset)
		if (just_daq)
			make/o/n=(m) sim_daq_avg=nan
		else
			make/o/n=(m,n_instr) sim_io_mat=nan
			make/o/n=(m) sim_ccn_ntot=nan, sim_daq_avg=nan, sim_daq_std=nan
			wlist+=",sim_ccn_ntot,sim_daq_avg,sim_daq_std,sim_io_mat"
			if (n_daq==0)		// Empty
			elseif (n_daq==1)	// Vector
				make/o/n=(m) sim_daq=nan, sim_dn=nan
				wlist+=",sim_daq,sim_dn"
			else				// Matrix
				make/o/n=(m,n_daq) sim_daq=nan, sim_dn=nan
				wlist+=",sim_daq,sim_dn"
			endif
			if (n_opc==1)		// Vector
				make/o/n=(m) sim_opc=nan
				wlist+=",sim_opc"
			elseif (n_opc>1)	// Matrix
				make/o/n=(m,n_opc) sim_opc=nan
				wlist+=",sim_opc"
			endif
		endif
	else
		wave sim_io_mat
		wave sim_ccn_ntot, sim_daq_avg, sim_daq_std
		wave sim_daq, sim_dn
		wave sim_opc
	endif
	// Put values to vectors
	variable i, j, k, flag
	for (i=0;i<n;i+=1)
		// Find current row
		FindValue/T=(0.1)/V=(tmp_data[i][0]) Scan_ID
		if (V_value==-1)
			print "Warning: scan ID "+num2str(tmp_data[i][0])+" not found!"
			continue
		endif
		j=V_value
		//
		//
		if (just_daq)	// Only the average droplet size
			sim_daq_avg[j]=tmp_data[i][n_instr+2]
		else			// Save everything
			// Data from the calculations
			sim_io_mat[j][0,n_instr-1]=tmp_data[i][q+1]		// e.g. SSinstr, Tmax, etc
			//
			// CCNp, DAVGo, SIGo
			sim_ccn_ntot[j]=tmp_data[i][n_instr+1]
			sim_daq_avg[j]=tmp_data[i][n_instr+2]
			sim_daq_std[j]=tmp_data[i][n_instr+3]
			//
			// NSEC x DN, NSEC x DAQ
			if (n_daq)
				sim_dn[j][]=tmp_data[i][n_instr+5+q]
				sim_daq[j][]=tmp_data[i][n_instr+5+n_daq+q]
			endif
			//
			if (n_opc)
				sim_opc[j][]=tmp_data[i][n_instr+6+2*n_daq+q]
			endif
		endif
	endfor
	Execute/q wlist
end
