/**
* \file get_mex_arguments.h
* \author basile.graf@epfl.ch
* 
* This file contains functions for getting parameters from MATLAB
*/


#ifndef GET_MEX_ARGUMENTS_H_DEF
#define GET_MEX_ARGUMENTS_H_DEF


#define PARAMNUM 10
#define STATENUM 12 //18 13:18=unused camera variables


/// Initialize integrator options to default
/**
* Initializes P.reltol and P.abstol to default values. \n
* The P.param array is set to zeros (not a valid default for meaningfull simulation, just to avoid any crash!!!)
* \param *P pointer to the parameters struct
*/
void initParameters(struct parameters *P);


/// Get integrator options parameters From MATLAB
/**
* Get solver settings from MATLAB. \n
* In the MATLAB prompt, a struct P can (must) be passed containing some or all of the 
* followings (if not set, default is used, exept for param): \n
* P.abstol:   Absolute tolerance (scalar (all the same) or vector of length 12) \n
* P.reltol:   Relative tolerance (scalar) \n
* P.param :   Parameters vector (diverse), MUST be given, see MATLAB model...
* \param nlhs Number of left hand side arguments in the MATLAB prompt
* \param *plhs[] Array of pointers to the left hand side arguments from MATLAB prompt
* \param nrhs Number of right hand side arguments in the MATLAB prompt
* \param *prhs[] Array of pointers to the right hand side arguments from MATLAB prompt
* \param *P pointer to the parameters struct (in C code)
*/
void get_Parameters(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[], struct parameters *P);


/// Get the initial (\f$ t=0 \f$) state vector (\f$ \vec{q} \f$) from MATLAB
/**
*  Copy \f$ \vec{q} \f$ at \f$ t=0 \f$ from MATLAB
* \param nlhs Number of left hand side arguments in the MATLAB prompt
* \param *plhs[] Array of pointers to the left hand side arguments from MATLAB prompt
* \param nrhs Number of right hand side arguments in the MATLAB prompt
* \param *prhs[] Array of pointers to the right hand side arguments from MATLAB prompt
* \return pointer to the double[12] state array (\f$ \vec{q} \f$)
*/
double* get_state(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]);

/// Free state vector (\f$ \vec{q} \f$) memory
/**
* Free the memory alocated by get_state()
* \param *state pointer to the double[12] state array (\f$ \vec{q} \f$)
*/
void delete_state(double *state);

/// Get flag from MATLAB (not used)
/**
* Get the flag parameter from MATLAB (NOT USED)
* \param nlhs Number of left hand side arguments in the MATLAB prompt
* \param *plhs[] Array of pointers to the left hand side arguments from MATLAB prompt
* \param nrhs Number of right hand side arguments in the MATLAB prompt
* \param *prhs[] Array of pointers to the right hand side arguments from MATLAB prompt
*/
char* get_flag(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]);

/// Get joystick options parameters From MATLAB
/**
* Get joystick settings from MATLAB. \n
* In the MATLAB prompt, a struct jP can be passed containing some or all of the 
* followings (if not set, default is used): \n
* jP.dirX:   Positive direction for axis X \n
* jP.dirY:   Positive direction for axis Y \n
* jP.dirZ:   Positive direction for axis Z \n
* jP.dirR:   Positive direction for axis R \n
* jP.doX:    Use joystick control or keyboard for axis X \n
* jP.doY:    Use joystick control or keyboard for axis Y \n
* jP.doZ:    Use joystick control or keyboard for axis Z \n
* jP.doR:    Use joystick control or keyboard for axis R \n
*/
void get_joystick_parameters(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[], struct joy_parameters *jP);


/// Get time parameter from MATLAB
/**
* Meaning: \n
* \f$ 0 \f$:   Simulate indefinitely without returning anything to MATLAB
* \f$ t_{end} \f$:   Simulate from \f$ t=0 \f$ to \f$ t=t_{end} \f$ and return simulation results to MATLAB
* \param nlhs Number of left hand side arguments in the MATLAB prompt
* \param *plhs[] Array of pointers to the left hand side arguments from MATLAB prompt
* \param nrhs Number of right hand side arguments in the MATLAB prompt
* \param *prhs[] Array of pointers to the right hand side arguments from MATLAB prompt
* \param *f_dat ODE solver data struct
*/
void get_time(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[], ODE_data *f_dat);

#endif