/**
* \file solid_objects.h
* \author basile.graf@epfl.ch
* 
* This file relates to all graphical objects. SolidObject is the 3D desription of an object like a boat hull. 
* This file contains also functions to draw navigation instruments on the screen. 
*/

#ifndef SOLID_OBJECTS_H_DEF
#define SOLID_OBJECTS_H_DEF

#include "../hydro_source/Datas.h"
#include "graphics_relative.h"

#include "math.h"
#include "mex.h"



/// 3D object description
/**
* Structure for the description of an object as a mesh.\n
*	N, M:				size of the mesh 2D array.\n
*	**X, **Y, **Z :		vertex coordinates.\n
*	**nX, **nY, **nZ :	normals to vertexes (for lighting).\n
*	**R, **G, **B :		vertex color.\n
*/
struct SolidObject
{
	int N, M; ///<matrices size

	float **X, **Y, **Z; ///<vertex coordinates

	float **nX, **nY, **nZ; ///<Vertex normals

	float **R, **G, **B; ///<vertex color
};




/// Creates a solid object
/** 
* Initializes memory for a 3D object \n
* Do not forget to call SolidObject_delete() for each object created once finished!!
* \param *so pointer to a SolidObject to be initialized
* \param N length of mesh 2D arrays (coordinates, normals, color)
* \param M height of mesh 2D arrays (coordinates, normals, color)
*/
void SolidObject_create(SolidObject *so, int N, int M);


/// Deletes a solid object
/**
* Free memory used by a 3D object \n
* To be used after every creation with SolidObject_create()
*/
void SolidObject_delete(SolidObject *so);


/// Computes the cross-product
/**
* \f$ v_c = v_a \times v_b  \f$
* \param *va : pointer to float array of 3 elements (\f$ v_a \f$)
* \param *vb : pointer to float array of 3 elements (\f$ v_b \f$)
* \param *vc : pointer to float array of 3 elements (\f$ v_c \f$)
*/
void cross_product(float *va, float *vb, float *vc);

/// Computes normal vector for one vertex
/**
* \f$ v_n = \frac{c}{|c|}\f$ \n
* \f$ c =  v_1 \times v_2 + v_2 \times v_3 + v_3 \times v_4 + v_4 \times v_1\f$
* \param *v1 : pointer to float array of 3 elements (\f$ v_1 \f$)
* \param *v2 : pointer to float array of 3 elements (\f$ v_2 \f$)
* \param *v3 : pointer to float array of 3 elements (\f$ v_3 \f$)
* \param *v4 : pointer to float array of 3 elements (\f$ v_4 \f$)
* \param *vn : pointer to float array of 3 elements (\f$ v_n \f$)
*/
void one_normal(float *v1, float *v2, float *v3, float *v4, float *vn);

/// Deletes the boat
/**
* Deletes all SolidObject objects used in drawing the boat using SolidObject_delete()
*/
void Boat_delete();

/// Computes the normals for a SolidObject
/**
* Once all X-, Y- Z- coordinates of a SolidObjects have been defined, 
* the vertex-normal vectors in the SolidObject can be automatically filled 
* by calling this function.
* \param *so: pointer to the SolidObject
*/
void SolidObject_compute_normals(SolidObject *so);

/// Creates and defines all boat related SolidObject objects
/**
* Create and fill all boat informations (coordinates,...) \n
* All hull coordinates are computed/defined herein. Have a look to the code!
* \param *d : pointer to the datas data-structure
*/
void Boat_create(datas *d);

/// Create and draw the sail
/**
* The sail is not a SolidObject. It is redefined each time it is drawn (its position
* relative to the boat can change). The code of this function contains OpenGL code.
* \param *d : pointer to the datas data-structure
*/
void Sail_create_and_draw(datas *d);

/// Draws relative wind vector
/**
* Draws a relative wind vector at the top of the mast. The code of this function contains OpenGL code.
* \param *d : pointer to the datas data-structure
*/
void RelativeWind_draw(datas *d);

/// Draws a "target instrument" on the screen
/**
* Draws a circular target with a cross inside.  The code of this function contains OpenGL code.
* \param x: cross position, \f$ -1.0 \leq x \leq 1.0\f$, clipped outside this range
* \param y: cross position, \f$ -1.0 \leq y \leq 1.0\f$, clipped outside this range
*/
void Target_draw(float x, float y);

/// Draws a "vertical slider instrument" on the screen
/**
* Draws a vertical slider.  The code of this function contains OpenGL code.
* \param x: slider position, \f$ -1.0 \leq x \leq 1.0\f$, clipped outside this range
*/
void VSlider_draw(float x);

/// Draws a "horizontal slider instrument" on the screen
/**
* Draws a horizontal slider.  The code of this function contains OpenGL code.
* \param x: slider position, \f$ -1.0 \leq x \leq 1.0\f$, clipped outside this range
*/
void HSlider_draw(float x);

/// Draws a girouette on the screen.
/**
* Draws a girouette instrument indicating relative wind direction as an on screen instrument.
* The code of this function contains OpenGL code.
* \param *d : pointer to the datas data-structure
*/
void Girouette_draw(datas *d);

/// Draws the boat.
/**
* Draws the boat once it has been created and defined by Boat_create(). This function uses
* calls to SolidObject_draw().  The code of this function contains OpenGL code.
*/
void Boat_draw();



#endif