%Author: Dominic Rfenacht, IVRG, EPFL, November 2013

%Purpose: Compute shadow masks for all images in a specified folder, using the three shadow detection methods, do the
%evaluation and create a HTML file with all the results. See the Readme.txt
%for more information.
%The two methods we use for comparison are described in:
%[1] J. Tian, L. Zhu, and Y. Tang. Outdoor shadow detection by combining tricolor attenuation and intensity. EURASIP Journal of Advanced Signal Processing, vol. 2012, p. 116, 2012.
%[2] R. Guo, Q. Dai, and D. Hoiem. Single-image shadow detection and removal using paired regions. CVPR. IEEE, pp. 20332040, 2011.

clear all

%Relative path to where the results will be stored
results_folder = 'Results';
%Relative path to folder where the ground truth images are stored
gt_folder = 'Ground_Truth';

%This allows to run the experiments only on a subpart of the images. "set"
%defines the folder where the images are stored. See below how
%images_folder is constructed.
set = '';

if(strcmp(set, ''))
    images_folder = [results_folder '/Images/'];
else
    images_folder = [results_folder '/Images/' set '/'];
end

%Select the methods. Possible choices are: 'Ours', 'Tian', and 'Guo'.
methods = {'Ours'};
%methods = {'Ours', 'Tian', 'Guo'};

%Create necessary directories
for j=1:length(methods)
    if(~exist([results_folder '/' methods{j}], 'dir'))
        disp(['Creating directory ' [results_folder '/' methods{j}] ' to save shadow maps...']);
        mkdir([results_folder '/' methods{j}]);
    end
end

if(~exist([results_folder '/Images/Thumbnails'], 'dir'))
    disp(['Creating directory ' [results_folder '/Images/Thumbnails'] ' to save thumbnails...']);
    mkdir([results_folder '/Images/Thumbnails']);
end

if(~exist([results_folder '/Ground_Truth/Thumbnails'], 'dir'))
    disp(['Creating directory ' [results_folder '/Ground_Truth/Thumbnails'] ' to save thumbnails...']);
    mkdir([results_folder '/Ground_Truth/Thumbnails']);
end

%Get all the images that are in "images_folder"
filenames_vis = dir(fullfile(['./' images_folder '*_vis.png']));
filenames_nir = dir(fullfile(['./' images_folder '*_nir.png']));

if(length(filenames_vis) ~= length(filenames_nir))
    error(['The number of visible and NIR images in the folder ' fullfile(['./' images_folder]) ' is not the same!']);
end

if(find(find(ismember(methods, 'Ours')==1)) == 1)
    %Apply our method for all images
    disp('Computing shadow masks for our method...')
    a = 14.0; %Determines the slope of the sigmoid function. Default: 14.0
    b = 0.5; %Inflection point of sigmoid. Default: 0.5
    thresh = 10.0; %thresh is the threshold used in the color to NIR computation. Default: 10.0
    useratios = 1; %Set this to zero if you don't want to use color to NIR ratios.
    gamma = 2.2; %To stretch the shadows in the tone mapping function before applying the sigmoid function f
    
    %Compute the shadow masks for all images that are is the "images_folder"
    for i=1:length(filenames_vis)
        disp(['Processing image ' filenames_vis(i).name ' (' num2str(i) '/' num2str(length(filenames_vis))  ')...']);
        
        vis = im2double(imread([images_folder filenames_vis(i).name]));
        try
            nir = rgb2gray(im2double(imread([images_folder filenames_nir(i).name])));
        catch
            nir = im2double(imread([images_folder filenames_nir(i).name]));
        end
        tic
        m1=min(min(vis)); M1=max(max(vis));
        m2=min(min(nir)); M2=max(max(nir));
        
        %Normalize the sensor responses
        vis(:,:,1)=(vis(:,:,1)-m1(:,:,1))/(M1(:,:,1)-m1(:,:,1));
        vis(:,:,2)=(vis(:,:,2)-m1(:,:,2))/(M1(:,:,2)-m1(:,:,2));
        vis(:,:,3)=(vis(:,:,3)-m1(:,:,3))/(M1(:,:,3)-m1(:,:,3));
        nir(:,:,1)=(nir(:,:,1)-m2(:,:,1))/(M2(:,:,1)-m2(:,:,1));
        
        %Make sure that everything is between 0 and 1.
        vis(vis > 1.0) = 1.0;
        vis(vis < 0.0) = 0.0;
        nir(nir > 1.0) = 1.0;
        nir(nir < 0.0) = 0.0;
        
        %Compute grayscale image of visible image
        L = (vis(:,:,1) + vis(:,:,2) + vis(:,:,3))/3.0;
        
        %Compute dark maps of visible and NIR image
        L = L.^(1/gamma);
        D_vis = f(L, a, b);
        nir = nir.^(1/gamma);
        D_nir = f(nir, a, b);
        
        %Compute shadow candidate map
        D = D_vis.*D_nir;
        
        %Compute color to NIR ratios (use this only for unprocessed images).      
        if(useratios == 1)
            [m,n,c] = size(vis);
            Tk = zeros(m,n,c);
            Tk(:,:,1) = vis(:,:,1)./nir;
            Tk(:,:,2) = vis(:,:,2)./nir;
            Tk(:,:,3) = vis(:,:,3)./nir;
            
            T = min(max(Tk, [], 3), thresh);
            
            %Normalize color to NIR ratio
            T = T./thresh;
            
            %Compute shadow map
            U = (1-D).*(1-T);
        else
            %If we don't use the color to NIR ratio, the shadow map will be
            %given by the shadow candidates.
            U = (1-D);
        end
        %Compute initial number of histogram bins
        [a1,b1]=size(U);
        HIST_LENGTH= 1.6*ceil(log2(a1*b1)+1);
        
        try
            [~, TT] = valley_hist(U, floor(HIST_LENGTH));
        catch
            disp(['Image ' filenames_vis(i).name ': Valley could not be determined. Using value of 0.5']);
            TT = 0.5;
        end
        theta = TT;
               
        %Compute shadow mask by thresholding at theta
        U_bin = U <= theta;
        t_ours(i) = toc;
        %Save resulting shadow mask
        namebase = filenames_vis(i).name(1:(regexp(filenames_vis(i).name, '_vis') - 1));
        %Write binary mask
        imwrite(U_bin, [results_folder '/Ours/' namebase '_ours.png']);
        %Write shadow map
        %imwrite(U, [results_folder '/Ours/map_' namebase '_ours.png']);
    end
    disp(['Done with our method... Total time is: ' num2str(sum(t_ours)) 's, average per image: ' num2str(sum(t_ours)/length(filenames_vis)) ' s, standard deviation: ' num2str(std(t_ours)) '.'])
end

%Tian et al.'s method:
%[1] J. Tian, L. Zhu, and Y. Tang. Outdoor shadow detection by combining tricolor attenuation and intensity. EURASIP Journal of Advanced Signal Processing, vol. 2012, p. 116, 2012.
if(find(find(ismember(methods, 'Tian')==1)))
    %Apply Tian's method
    disp('Computing shadow masks for Tians method...')
    for i=1:length(filenames_vis)
        disp(['Processing image ' filenames_vis(i).name ' (' num2str(i) '/' num2str(length(filenames_vis))  ')...']);
        img = imread([images_folder filenames_vis(i).name]);
        img = uint8(im2double(img).*255);
        %Convert to grayscale image y
        y=rgb2gray(img);
        tic
        img1 = img(:,:,1);
        img2 = img(:,:,2);
        img3 = img(:,:,3);
        meanR = mean2(img1);
        meanG = mean2(img2);
        meanB = mean2(img3);
        R1 = img1 > meanR;
        G1 = img2 > meanG;
        B1 = img3 > meanB;
        meanR = mean2(nonzeros(R1));
        meanG = mean2(nonzeros(G1));
        meanB = mean2(nonzeros(B1));
        [m n] = sort([1.31*meanR/meanB 1.19*meanG/meanB 1]);
        
        P = eval(strcat('img',num2str(n(3))));
        Q = eval(strcat('img',num2str(n(1))));
        %Compute TAM image x
        x = P - Q;
        
        %Initialize alpha as mean(y)/mean(x)
        alpha = ceil(mean2(y)/mean2(x));
        if alpha>10;
            alpha = alpha/2;
        end
        
        beta = 1;
        z = alpha*x + beta*y;
        [Value T alpha beta] = jisuan1(z, x, y);
        for k=1:100
            z = exp(alpha/beta)*x+y;
            [Vnew Tnew alpha beta] = jisuan1(z, x, y);
            if (Vnew > Value)&(abs(T - Tnew) > 0)
                Value = Vnew;
                T = Tnew;
            else
                break;
            end
        end
        %Threshold the image
        ax = (z < Tnew);
        t_tian(i) = toc;
        
        %Save the shadow mask
        namebase = filenames_vis(i).name(1:(regexp(filenames_vis(i).name, '_vis') - 1));
        %Save resulting shadow mask
        imwrite(ax, [results_folder '/Tian/' namebase '_tian.png']);
        %imwrite(z, [results_folder '/Tian/map_' namebase '_tian.png']);
    end
    disp(['Done with Tians method... Total time is: ' num2str(sum(t_tian)) 's, average per image: ' num2str(sum(t_tian)/length(filenames_vis)) ' s, standard deviation: ' num2str(std(t_tian)) '.'])
end

%Guo et al.'s method:
%[2] R. Guo, Q. Dai, and D. Hoiem. Single-image shadow detection and removal using paired regions. CVPR. IEEE, pp. 20332040, 2011.
if(find(find(ismember(methods, 'Guo')==1)))
    %Apply Guo et al.'s method
    disp('Computing shadow masks for Guos method...')
    addpath(genpath('other1_cvpr2011'))
    opt = {};
    opt.dir = 'other1_cvpr2011/data/';
    
    %Apply Guo's method
    for i=1:length(filenames_vis)
        disp(['Processing image ' filenames_vis(i).name ' (' num2str(i) '/' num2str(length(filenames_vis))  ')...']);
        %vis = im2double(imread([images_folder filenames_vis(i).name]));
        
        opt.fn = [filenames_vis(i).name];
        opt.basedir = images_folder;
        opt.save = 0;
        opt.binaryClassifier = 'models/model_pair.mat';
        opt.unaryClassifier = 'models/unary_model.mat';
        %opt.binaryClassifier = 'models/model_pair_our.mat';
        %opt.unaryClassifier = 'models/unary_model_our.mat';
        opt.resize = 0;
        opt.adjecent = 0;
        opt.pairwiseonly = 0;
        opt.linearize = 0;
        tic
        mask = findshadowcut_cross(opt);
        
        t_guo(i) = toc;
        
        %Save resulting shadow mask
        namebase = filenames_vis(i).name(1:(regexp(filenames_vis(i).name, '_vis') - 1));
        imwrite(mask, [results_folder '/Guo/' namebase '_guo.png']);
    end
    disp(['Done with Guos method... Total time is: ' num2str(sum(t_guo)) 's, average per image: ' num2str(sum(t_guo)/length(filenames_vis)) ' s, standard deviation: ' num2str(std(t_guo)) '.'])
end
%%
%Create HTML file containing all the results
horizontal_screen_resolution = 1400;
%Compute size of one image in HTML file based on screen resolution and
%number of methods tested
iSize = num2str(floor((horizontal_screen_resolution-((2+length(methods))*25))/(3+length(methods))));%num2str(300); %Width of one image in the HTML file for comparison (generated at the end of this code)

disp('Computing statistics and creating HTML file...')
htmlStr = [];
%Write HTML header
htmlStr = sprintf('%s <div style="font-family:Lucida Grande,Helvetica,Arial,sans-serif;font-size: 12px;" align="center" >\n<table BORDER=2 CELLPADDING=5 RULES=GROUPS FRAME=BOX>\n<THEAD>\n<tr align="center">\n<th>Original Image </th> <th>NIR Image </th> <th> Ground Truth </th> ', htmlStr);

%Create column titles for each method
for j=1:length(methods)
    htmlStr = sprintf('%s <th> %s </th> ', htmlStr, methods{j});
end
%End the header line
htmlStr = sprintf('%s </tr>\n</THEAD>\n', htmlStr);

prec = cell(length(methods), length(filenames_vis));
tpr = cell(length(methods), length(filenames_vis));
fpr = cell(length(methods), length(filenames_vis));
thresh = cell(length(methods), length(filenames_vis));

for i=1:length(filenames_vis)
    disp(i)
    foo = textscan(filenames_vis(i).name,'%s', 'delimiter', '.');
    namebase = foo{1}{1};
    foo = textscan(namebase,'%s', 'delimiter', '_');
    number = foo{1}{2};
    namebase = filenames_vis(i).name(1:(regexp(filenames_vis(i).name, '_vis') - 1));
    %Load the groundtruth as well as shadow masks in "directory"
    try
        if(strcmp(namebase(end), 'f') || strcmp(namebase(end), 'i'))
            number = number(1:end-1);
        end
        
        gt = rgb2gray(im2double(imread([results_folder '/' gt_folder '/img_' number '_gt.png'])));
    catch
        movefile([results_folder '/' gt_folder '/img_' number '.png'], [results_folder '/' gt_folder '/img_' number '_gt.png']);
        gt = rgb2gray(im2double(imread([results_folder '/' gt_folder '/img_' number '_gt.png'])));
    end
    
    %Make sure the ground truth is binary
    gt(gt >= 0.5) = 1.0;
    gt(gt < 0.5) = 0.0;
    [m, n] = size(gt);
    
    original_filename_vis = ['./Images/' filenames_vis(i).name];
    original_filename_nir = ['./Images/' filenames_nir(i).name];
    gt_filename = ['./Ground_Truth/img_' number '_gt.png'];
    
    %Thumbnails
    original_filename_vis_thumb = ['./Images/Thumbnails/' filenames_vis(i).name];
    original_filename_nir_thumb = ['./Images/Thumbnails/' filenames_nir(i).name];
    gt_filename_thumb = ['./Ground_Truth/Thumbnails/img_' number '_gt.png'];
    
    
    %Create thumbnail for visible image
    toto = im2double(imread([results_folder '/Images/' filenames_vis(i).name]));
    imwrite(imresize(toto, [NaN str2num(iSize)]), [results_folder '/Images/Thumbnails/' filenames_vis(i).name]);
    
    %Create thumbnail for NIR image
    toto = im2double(imread([results_folder '/Images/' filenames_nir(i).name]));
    imwrite(imresize(toto, [NaN str2num(iSize)]), [results_folder '/Images/Thumbnails/' filenames_nir(i).name]);
    
    
    %Create thumbnail for Ground Truth mask
    toto = im2double(imread([results_folder  '/Ground_Truth/img_' number '_gt.png']));
    toto = imresize(toto, [NaN str2num(iSize)]);
    toto(toto > 0.5) = 1.0;
    toto(toto <= 0.5) = 0.0;
    imwrite(toto, [results_folder '/Ground_Truth/Thumbnails/img_' number '_gt.png']);
    
    %Write first part of table row
    htmlStr = sprintf('%s\n<TBODY>\n<tr align="center"> <td><a href="%s"><img src="%s" width=%s></a></td>  <td><a href="%s"><img src="%s" width=%s></a></td>  <td><a href="%s"><img src="%s" width=%s></a></td>', htmlStr, original_filename_vis, original_filename_vis_thumb, iSize, original_filename_nir, original_filename_nir_thumb, iSize, gt_filename, gt_filename_thumb, iSize);
    
    for j=1:length(methods)
        %Load mask and resize it to have the same size as the ground truth map
        %keyboard
        mask = im2double(imread([results_folder '/' methods{j} '/' namebase '_' lower(methods{j}) '.png']));
        %shadow_map = im2double(imread([results_folder '/' methods{j} '/map_' namebase '_' lower(methods{j}) '.png']));
        [m1 n1] = size(mask);
        [m2 n2] = size(gt);
        if(m1 ~= m2 || n1 ~=n2)
            mask = imresize(mask, [m2, n2]);
            %shadow_map = imresize(shadow_map, [m2, n2]);
        end
        %Make sure the mask is binary
        mask(mask >= 0.5) = 1.0;
        mask(mask < 0.5) = 0.0;
        
        %Compute accuracy between the mask of method compared to the ground
        %truth
        [tp fp tn fn res(i,j)] = compute_accuracy(mask, gt);
        mcc(i,j) = compute_mcc(tp, fp, tn, fn);
        
        %Save result
        if(~exist([results_folder '/' methods{j} '/Thumbnails/'], 'dir'))
            disp('Creating directory to save results...');
            mkdir([results_folder '/' methods{j} '/Thumbnails/']);
        end
        
        %Generate mask and mask thumbnail filenames
        mask_filename{j} = ['./' methods{j} '/' namebase '_' lower(methods{j}) '.png'];
        mask_filename_thumb{j} = ['./' methods{j} '/Thumbnails/' namebase '_' lower(methods{j}) '.png'];
        
        %Save thumbnail of mask
        toto = imresize(mask, [NaN str2num(iSize)]);
        toto(toto > 0.5) = 1.0;
        toto(toto <= 0.5) = 0.0;
        imwrite(toto, [results_folder '/' methods{j} '/Thumbnails/' namebase '_' lower(methods{j}) '.png']);
        
        %Continue HTML string
        htmlStr = sprintf('%s <td><a href="%s"><img src="%s" width=%s></a></td>', htmlStr, mask_filename{j}, mask_filename_thumb{j}, iSize);
    end
    
    htmlStr = sprintf('%s </tr> \n', htmlStr);
    htmlStr = sprintf('%s <tr align="center"> <td> %s </td> <td> %s </td> <td> Ground Truth </td> ', htmlStr, filenames_vis(i).name, filenames_nir(i).name);
    
    for j=1:length(methods)
        htmlStr = sprintf('%s <td> Acc= %s, MCC= %s </td>', htmlStr, num2str(res(i,j)), num2str(mcc(i,j)));
    end
    htmlStr = sprintf('%s </tr>\n</TBODY> \n', htmlStr);
end

%Begin HTML footer
htmlStr = sprintf('%s\n<TFOOT>\n<tr> <td> %s images in total. </td> <td>  </td> <td>  </td> ', htmlStr, num2str(length(filenames_vis)));

%Compute statistics
avg = zeros(length(methods), 1);
stdev = zeros(length(methods), 1);
mcc_mean = zeros(length(methods), 1);
mcc_stddev = zeros(length(methods), 1);
for j=1:length(methods)
    avg(j) = mean(res(:,j));
    stdev(j) = std(res(:,j));
    mcc_mean(j) = mean(mcc(:,j));
    mcc_stddev(j) = std(mcc(:,j));
    htmlStr = sprintf('%s <td>  Average: %s <br> Std dev.: %s <br> Mean MCC : %s <br> Std dev MCC: %s </td>', htmlStr, num2str(avg(j)), num2str(stdev(j)), num2str(mcc_mean(j)), num2str(mcc_stddev(j)));
end

clear res;
htmlStr = sprintf('%s </tr>\n</TFOOT>\n', htmlStr);

%End table
htmlStr = sprintf('%s\n</table>\n</div> \n', htmlStr);

htmlStr = sprintf('<!DOCTYPE HTML>\n\n<html>\n<head>\n</head>\n\n<body>\n%s\n</body>\n</html>', htmlStr);

%Write HTML file to disk
htmlFilename = [results_folder '/results_' lower(set) '.html'];

fid = fopen(htmlFilename, 'w');
fprintf(fid, '%s', htmlStr);
fclose(fid);

disp('All done...')
