function hdrimage = mergeHDR(images, texp, low_threshold, high_threshold)
% function mergeHDR
% merge several images into one HDR image from
% 
% IN:
%   images : CFA images in a 3D matrix: (:,:,image_number)
%   texp : exposure time vector, order must correspond to images order
%   low_threshold and high_threshold: pixel that are not taken into account
% OUT:
%   hdrimage : merged HDR image
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% -----------------------------------------------------------------------
% Digital Camera Workflow for HDR images using a model of retinal
% processing
% Written by Daniel Tamburrino, David Alleysson and Laurence Meylan
% Copyright (C) 2007-2008 Laboratory of Audiovisual Communications (LCAV), 
% Ecole Polytechnique Federale de Lausanne (EPFL), 
% CH-1015 Lausanne, Switzerland 
% 
% This program is free software; you can redistribute it and/or modify it 
% under the terms of the GNU General Public License as published by the 
% Free Software Foundation; either version 2 of the License, or (at your 
% option) any later version. This software is distributed in the hope that 
% it will be useful, but without any warranty; without even the implied 
% warranty of merchantability or fitness for a particular purpose. 
% See the GNU General Public License for more details 
% (enclosed in the file GPL). 
%
% Latest modifications: November 28, 2007, by Daniel Tamburrino

global DISPLAY

[nrows ncolumns npictures] = size(images);
hdrimage = zeros(nrows, ncolumns);

% FUSION
%find indice of darkest and brightest image
darkest=find(texp==min(texp));
brightest=find(texp==max(texp));

%Create fusion mask to eliminate under-exposed or over-exposed pixels
max_value=max(images(:));
mask = not(or(images > high_threshold*max_value, images < low_threshold*max_value)); %create normal mask
% find location where there is no pixel value
zero_mask = sum(mask,3);
zero_mask = ceil(zero_mask/max(zero_mask(:))); % mask with zero or ones. zero means pixel location with no value, i.e. 0.
zero_mask = not(zero_mask); %inverse mask: 1 indicate missing pixel value location
% create masks for missing pixels (only), in shadows and highlights
mask_shadows= zero_mask.*not(images(:,:,brightest) > high_threshold*max_value);
mask_highlight= zero_mask.*not(images(:,:,darkest) < low_threshold*max_value);
% add to original mask to take into account the missing pixels
mask(:,:,darkest)=mask(:,:,darkest) + mask_highlight;
mask(:,:,brightest)=mask(:,:,brightest) + mask_shadows;


Nnotsat = max(sum(mask, 3), 1); % to take into account the number of time a pixel is counted

images = images.* single(mask); % single precision is enough and uses less memory
for picture=1:npictures
    hdrimage = hdrimage + images(:,:,picture)./texp(picture)* max(texp);
end
%
hdrimage = hdrimage./Nnotsat; % to take into account the number of time a pixel is counted
max(hdrimage(:))
hdrimage=hdrimage/max(hdrimage(:)); %normalization

% Display map of non-zero pixels
if DISPLAY
    pixmap=sum(mask,3);
    pixmap=min(pixmap,1);
    figure, imshow(pixmap)
    title('Pixel map for HDR image creation.')
    fprintf('black pixels are not taken into account, i.e. zero value \n')
    for i=1:npictures
        figure
        imshow(mask(:,:,i))
        title(['Mask ' num2str(i)])
    end
end